#!/bin/bash

# Docker install script for CLEV2ER SII

# Capture the start time
start_time=$(date +%s)

VERSION=0.1.3

# Package sizes in GB
TDS_SIZE="5.5GB"

# Colors
GREEN='\033[0;32m'
RED='\033[0;31m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

clear

echo ""
echo "--------------------------------------------------------------------------"
echo -e "Installation script for ${BLUE}CLEV2ER Sea Ice and Iceberg ${RED}(docker install)${NC}"
echo "Software version: $VERSION"
echo "--------------------------------------------------------------------------"
echo ""

# Detect platform (Linux or macOS)
PLATFORM=$(uname)

# Function to verify SHA256 checksum
verify_checksum() {
    local file=$1
    local checksum_file=$2

    if [[ "$PLATFORM" == "Linux" ]]; then
        sha256sum -c "$checksum_file"
    elif [[ "$PLATFORM" == "Darwin" ]]; then
        shasum -a 256 -c "$checksum_file"
    else
        echo "Unsupported platform: $PLATFORM"
        exit 1
    fi
}



# Function to ensure valid input
# usage: ensure_valid_response "Y N" "Enter Y or N" ANSWER
ensure_valid_response() {
    local options="$1"
    local prompt_message="$2"
    local response_var="$3"
    while true; do
        read -p "$prompt_message" RESPONSE
        RESPONSE=${RESPONSE:-$DEFAULT_RESPONSE}
        for option in ${options//,/ }; do
            if [[ "$RESPONSE" == "$option" ]]; then
                eval $response_var="'$RESPONSE'"
                return 0
            fi
        done
        echo "Invalid response. Please enter one of: $options."
    done
}

# Function to check if pv is installed
check_pv_installed() {
    if command -v pv &> /dev/null; then
        echo "yes"
    else
        echo "no"
    fi
}

# Check if pv is installed
PV_INSTALLED=$(check_pv_installed)

# Function to check if a command exists
command_exists() {
    command -v "$1" >/dev/null 2>&1
}

# Check if Docker is installed
if ! command_exists docker; then
    echo -e "${RED}Docker is not installed. Please install docker on your system before proceeding.${NC}"
    exit 1
fi

# Check if Docker daemon is running
if ! pgrep -x "dockerd" >/dev/null 2>&1; then
    echo -e "${RED}Docker daemon is not running. Please install docker on your system before proceeding.${NC}"
    exit 1
fi

# Check if the current user can run Docker commands
if ! docker info >/dev/null 2>&1; then
    echo -e "${RED}The current user does not have permission to run Docker commands.${NC}"
    echo "Please add the current user to the docker group."
    exit 1
fi

echo -e "${GREEN}Docker is installed & available for the current user.${NC}"
echo ""

echo -e "${BLUE}Step 1: Select theme to install (seaice, iceberg, or both)${NC}"
#DEFAULT_RESPONSE='both'
echo ""
ensure_valid_response "seaice iceberg both" "Enter seaice, iceberg or both :): " THEME

# Conditional assignment
if [[ $THEME == "seaice" ]]; then
    THEME_STR="Sea Ice"
    THEME_TAG="si"
    TDS_SIZE="0.01GB"
    DOCKER_SIZE_WITH_TDS="893MB"
    DOCKER_STANDARD_SIZE="893MB"
elif [[ $THEME == "iceberg" ]]; then
    THEME_STR="Iceberg"
    THEME_TAG="ib"
    TDS_SIZE="0.01GB"
    DOCKER_SIZE_WITH_TDS="893MB"
    DOCKER_STANDARD_SIZE="893MB"
elif [[ $THEME == "both" ]]; then
    THEME_STR="Sea Ice and Iceberg"
    THEME_TAG="siib"
    DOCKER_SIZE_WITH_TDS="893MB"
    DOCKER_STANDARD_SIZE="893MB"
    TDS_SIZE="0.01GB"
fi

echo ""
echo -e "${GREEN}$THEME_STR selected${NC}"
echo ""

echo -e "${BLUE}Step 2: Select the CLEV2ER SII Docker image from 2 options${NC}"
echo "----------------------------------------------------------------"
echo ""
echo -e "1) ${RED}with${NC} large TDS inside image"
echo -e "2) ${RED}without${NC} large TDS"
echo ""

ensure_valid_response "1 2" "Select option (1/2) : " OPTION


if [ $OPTION -eq 2 ]; then
    image_file="clev2er_sii_${THEME_TAG}-$VERSION.docker.tar.gz"
    SIZE=${DOCKER_STANDARD_SIZE}
else
    image_file="clev2er_sii_${THEME_TAG}_with_tds-${VERSION}.docker.tar.gz"
    SIZE=${DOCKER_SIZE_WITH_TDS}
fi

echo ""

# Prompt for download directory
echo -e "${BLUE}Step 3: Download docker package: $image_file${NC}"
echo "-------------------------------------------------------------------------"
echo -e "${BLUE}Enter the ${RED}temporary${BLUE} download directory path for docker file [${SIZE}]${NC}"
echo "This is where the docker image file is temporarily downloaded to."
echo "It will be removed (cleaned) from this directory once loaded into Docker"
read -p "Enter path [or press Enter for default of current dir $(pwd) ]: " DOWNLOAD_DIR
DOWNLOAD_DIR=$(readlink -f "${DOWNLOAD_DIR:-$(pwd)}")

# Ensure the download directory exists
if [ ! -d "$DOWNLOAD_DIR" ]; then
    echo "Directory $DOWNLOAD_DIR does not exist. Creating it..."
    mkdir -p "$DOWNLOAD_DIR"
    if [ $? -ne 0 ]; then
        echo -e "${RED}Failed to create directory $DOWNLOAD_DIR. Exiting.${NC}"
        exit 1
    fi
fi

cd $DOWNLOAD_DIR

echo ""
echo -e "Securely downloading CLEV2ER SII $THEME_STR v$VERSION ${BLUE}docker package${NC} to $DOWNLOAD_DIR ..."
echo ""
curl -f -S --progress-bar -O https://www.cpom.ucl.ac.uk/downloads/clev2er_sii/$image_file
if [ $? -ne 0 ]; then
    echo -e "${RED}Failed to download CLEV2ER SII $THEME_STR v$VERSION! : $image_file ${NC}"
    exit 1
fi
echo -e "${GREEN}Download completed successfully.${NC}"

echo ""
echo "Securely Downloading CLEV2ER SII $THEME_STR v$VERSION SHA256 checksum..."
echo ""

curl -f -S --progress-bar -O https://www.cpom.ucl.ac.uk/downloads/clev2er_sii/${image_file}.sha256
if [ $? -ne 0 ]; then
    echo -e "${RED}Failed to download SHA256 checksum for CLEV2ER SII $THEME_STR docker v$VERSION! ${image_file}.sha256 ${NC}"
    exit 1
fi
echo -e "${GREEN}Download of SHA256 checksum completed successfully.${NC}"

echo ""
echo "Verify integrity of $image_file using SHA256 checksum..."

if verify_checksum "$image_file" "${image_file}.sha256"; then
    echo -e "${GREEN}SHA256 checksum verification succeeded.${NC}"
    # Continue with further processing
else
    echo -e "${RED}SHA256 checksum verification failed!${NC}"
    exit 1
fi

# Remove the checksum file
rm ${image_file}.sha256

echo ""
echo -e "${BLUE}Loading the docker image [only required once]...${NC}"
docker load -i $DOWNLOAD_DIR/$image_file
# Tag the image with 'latest'

docker tag clev2er/${THEME_TAG}:$VERSION clev2er/${THEME_TAG}:latest

if [ $? -eq 0 ]; then
    echo -e "${GREEN}Docker image loaded successfully.${NC}"
else
    echo -e "${RED}Failed to load Docker image.${NC}"
    exit 1
fi

rm -f $DOWNLOAD_DIR/$image_file

if [ $OPTION -eq 1 ]; then
    echo ""
    echo -e "${GREEN}Installation complete${NC}"
    echo ""
    echo "You can test the docker install with:"
    echo ""
    echo -e "${BLUE}docker run${NC} --rm clev2er/${THEME_TAG}:latest run_chain -v"
    echo ""
    echo "You can setup for an interactive shell with:"
    echo ""
    echo -e "${BLUE}docker run${NC} --rm -it clev2er/${THEME_TAG}:latest /bin/bash"
    echo ""
    exit 0
fi

#----------------------------------------------------------------------------------------------------------

echo ""
echo -e "${BLUE}Step 4: Optional TDS/ADF Installation${NC}"
echo "---------------------------------------------"

echo -e "${BLUE}Do you want to install the optional large test data set/ADF package for $THEME_STR?${NC}"
echo "** This is ${TDS_SIZE} in size **"
echo "This is recommended in order to run the full GPP chains unless you provide your own ADF files"
ensure_valid_response "Y N" "Y/N : " ANSWER
if [[ "$ANSWER" == 'N' ]]; then
    echo -e "${GREEN}Installation of the CLEV2ER Sea Ice and Iceberg GPP software completed without external TDS/ADF package${NC}"
    echo -e "${RED}Note, that for full GPP functionality, all ADF locations will be required to be configured${NC}"
    echo "You can test the docker install with:"
    echo ""
    echo -e "${BLUE}docker run${NC} --rm clev2er/${THEME_TAG}:latest run_chain -v"
    echo ""
    echo "You can setup for an interactive shell with:"
    echo ""
    echo -e "${BLUE}docker run${NC} --rm -it clev2er/${THEME_TAG}:latest /bin/bash"
    exit 0
fi

echo ""
echo -e "${BLUE}Enter directory to download the large TDS/ADF package${NC} : ${TDS_SIZE} for version $VERSION"
echo "Note this is where the TDS package will be temporarily downloaded to (not installed) "
echo "Downloaded .tar.gz files will be cleaned from this directory after installation. "
read -p "Enter path for TDS download (or press ENTER for default: $DOWNLOAD_DIR) : " DOWNLOAD_DIR_tmp
DOWNLOAD_DIR=$(readlink -f "${DOWNLOAD_DIR_tmp:-$DOWNLOAD_DIR}")

# Ensure the download directory exists
if [ ! -d "$DOWNLOAD_DIR" ]; then
    echo "Directory $DOWNLOAD_DIR does not exist. Creating it..."
    mkdir -p "$DOWNLOAD_DIR"
    if [ $? -ne 0 ]; then
        echo -e "${RED}Failed to create directory $DOWNLOAD_DIR. Exiting.${NC}"
        exit 1
    fi
fi

echo ""
echo "Downloading TDS for version $VERSION .."
echo ""

cd $DOWNLOAD_DIR
curl -f -S --progress-bar -O \
https://www.cpom.ucl.ac.uk/downloads/clev2er_sii/testdata_external_${THEME_TAG}_$VERSION.tar.gz
if [ $? -ne 0 ]; then
    echo -e "${RED}Failed to download testdata_external_$VERSION.tar.gz!${NC}"
    exit 1
fi

echo "Downloading TDS SHA256 checksum for $VERSION .."

curl -f -S --progress-bar -O \
https://www.cpom.ucl.ac.uk/downloads/clev2er_sii/testdata_external_${THEME_TAG}_$VERSION.tar.gz.sha256
if [ $? -ne 0 ]; then
    echo -e "${RED}Failed to download testdata_external_$VERSION.tar.gz.sha256!${NC}"
    exit 1
fi
echo -e "${GREEN}TDS download successful${NC}"

echo "Verifying SHA256 checksum..."
if sha256sum -c testdata_external_${THEME_TAG}_$VERSION.tar.gz.sha256; then
    echo -e "${GREEN}SHA256 checksum verification succeeded.${NC}"
    # Continue with further processing
else
    echo  -e "${RED}SHA256 checksum verification failed!${NC}"
    exit 1
fi

rm testdata_external_${THEME_TAG}_$VERSION.tar.gz.sha256

# Prompt for installation directory
SOFT_INSTALL_DIR=$INSTALL_DIR
echo " "
echo "TDS/ADF package will be unpacked to <TDS_INSTALL_DIR>/testdata_external"
echo -e "${BLUE}Enter the TDS installation directory <TDS_INSTALL_DIR> path${NC}"

read -p "Enter path [or press ENTER to accept default: $(pwd)]: " TDS_INSTALL_DIR
TDS_INSTALL_DIR=$(readlink -f "${TDS_INSTALL_DIR:-$(pwd)}")

if [ "$TDS_INSTALL_DIR" == "${SOFT_INSTALL_DIR}/clev2er_sii-$VERSION" ]; then
    TDS_IN_STANDARD_LOCATION='yes'
fi

# Ensure the installation directory exists
if [ ! -d "$TDS_INSTALL_DIR" ]; then
    echo "Directory $TDS_INSTALL_DIR does not exist. Creating it..."
    mkdir -p "$TDS_INSTALL_DIR"
    if [ $? -ne 0 ]; then
        echo -e "${RED}Failed to create directory $TDS_INSTALL_DIR. Exiting.${NC}"
        exit 1
    fi
fi

echo ""
if [ -d $TDS_INSTALL_DIR/testdata_external ]; then
    echo "TDS already installed in $TDS_INSTALL_DIR/testdata_external"
    ensure_valid_response "Y N" "Replace existing TDS installation? (Y/N): " ANSWER
    if [[ "$ANSWER" == 'N' || "$ANSWER" == 'n' ]]; then
        echo "Exiting installation as TDS already installed"
        exit 0
    fi
fi

# Continue with the installation process
echo " "
echo "Installing TDS to $TDS_INSTALL_DIR/testdata_external"
# Add your installation commands here

echo "Unpacking TDS v$VERSION..."

if [ "$PV_INSTALLED" == "yes" ]; then
    pv $DOWNLOAD_DIR/testdata_external_${THEME_TAG}_$VERSION.tar.gz | tar -xz -C $TDS_INSTALL_DIR && rm $DOWNLOAD_DIR/testdata_external_${THEME_TAG}_$VERSION.tar.gz
else
    tar -xzvf $DOWNLOAD_DIR/testdata_external_$VERSION.tar.gz \
-C $TDS_INSTALL_DIR && rm $DOWNLOAD_DIR/testdata_external_${THEME_TAG}_$VERSION.tar.gz
fi

echo ""
echo -e "${GREEN}Installation complete${NC}"

echo -e "${BLUE}Summary${NC}"
echo "Docker image loaded as : clev2er/${THEME_TAG}:$VERSION"
echo "Docker image also tagged as : clev2er/${THEME_TAG}:latest"
echo "TDS installed to $TDS_INSTALL_DIR/testdata_external"
echo "TDS can mounted within the docker image using :"
echo "docker run --rm -it -v $TDS_INSTALL_DIR/testdata_external:/app/testdata_external clev2er/${THEME_TAG}:latest /bin/bash"
exit 0
