#!/usr/bin/env bash

# Environment Installation script for CLEV2ER SII framework) on linux/macos

set -e  # Exit on any error

setup_and_run_file=./activate.sh

export CLEV2ER_BASE_DIR=$PWD

# Generate the setup_and_run.sh script
echo "#!/usr/bin/env bash" > $setup_and_run_file
echo "" >> $setup_and_run_file

echo "# Check if the script is being sourced or executed" >> $setup_and_run_file
echo 'if [[ "${BASH_SOURCE[0]}" == "${0}" ]]; then' >> $setup_and_run_file
echo '    echo "ERROR: This script must be sourced, not executed!"' >> $setup_and_run_file
echo '    echo "Please run: source ./ct_activate.sh"' >> $setup_and_run_file
echo '    exit 1' >> $setup_and_run_file
echo "fi" >> $setup_and_run_file

echo "# Combined setup and run script for CLEV2ER SII" >> $setup_and_run_file
echo 'old_opts=$(set +o | grep errexit)' >> $setup_and_run_file
echo 'export PATH="$HOME/.local/bin:$PATH"' >> $setup_and_run_file
echo "" >> $setup_and_run_file
echo "if command -v conda &>/dev/null; then" >> $setup_and_run_file
echo '    if [[ -n "${CONDA_SHLVL}" && "${CONDA_SHLVL}" -gt 0 ]]; then' >> $setup_and_run_file
echo "        conda deactivate || true" >> $setup_and_run_file
echo "    fi" >> $setup_and_run_file
echo "fi" >> $setup_and_run_file
echo "unset CONDA_PREFIX" >> $setup_and_run_file
echo "unset CONDA_DEFAULT_ENV" >> $setup_and_run_file
echo "unset VIRTUAL_ENV" >> $setup_and_run_file

echo "# Activate Poetry virtual environment" >> $setup_and_run_file
echo "VENV_PATH=\$(poetry env info --path)" >> $setup_and_run_file
echo "if [ -z \"\$VENV_PATH\" ]; then" >> $setup_and_run_file
echo "    echo \"Poetry virtual environment not found. Did you set it up?\"" >> $setup_and_run_file
echo "    exit 1" >> $setup_and_run_file
echo "fi" >> $setup_and_run_file
echo "source \"\$VENV_PATH/bin/activate\"" >> $setup_and_run_file
echo "" >> $setup_and_run_file

# Export environment variables to the script
echo "export CLEV2ER_BASE_DIR=$PWD" >> $setup_and_run_file
echo "export PYTHONPATH=$CLEV2ER_BASE_DIR/src" >> $setup_and_run_file
echo "export PATH=${CLEV2ER_BASE_DIR}/src/clev2er/tools:\${PATH}" >> $setup_and_run_file

# Add path existence checks
# Add path and environment variable existence checks
echo "" >> $setup_and_run_file

# Set ulimit
echo "" >> $setup_and_run_file
echo "ulimit -n 8192" >> $setup_and_run_file

# Notify user the environment is ready
echo "" >> $setup_and_run_file
echo "eval \"$old_opts\"" >> $setup_and_run_file

echo "echo \"Environment setup complete. You are now in the CLEV2ER SII Poetry virtual environment.\"" >> $setup_and_run_file

# Ensure the output script is executable
chmod +x $setup_and_run_file

# Install Python and dependencies
conda_used=0

if command -v python3.12 &>/dev/null; then
    echo "Python 3.12 is already installed."
else
    if command -v conda &>/dev/null; then
        echo "Conda is available, creating Python 3.12 environment..."
        conda create -n py312 python=3.12 -y
        conda_used=1
    else
        echo "Installing Miniconda..."
        curl -L https://repo.anaconda.com/miniconda/Miniconda3-latest-Linux-x86_64.sh -o miniconda.sh
        bash miniconda.sh -b -p "$HOME/miniconda"
        rm miniconda.sh
        export PATH="$HOME/miniconda/bin:$PATH"
        conda init
        conda create -n py312 python=3.12 -y
        conda_used=1
    fi
fi

if [ $conda_used -eq 1 ]; then
    # If you had activated the conda env, grab its path, then deactivate it.
    source "$(conda info --base)/etc/profile.d/conda.sh"
    conda activate py312
    CONDA_ENV_PATH="$CONDA_PREFIX"  # e.g. /home/ctopr/.conda/envs/py312

    echo "Conda environment activated at $CONDA_ENV_PATH. Now deactivating it..."
    conda deactivate

    # Unset environment markers that might signal an active virtual env
    unset CONDA_DEFAULT_ENV
    unset CONDA_PREFIX
    unset VIRTUAL_ENV

    # Add the Python from the conda env to PATH so Poetry can find python3.12
    export PATH="$CONDA_ENV_PATH/bin:$PATH"
fi

# Install Poetry (if not already installed)
curl -sSL https://install.python-poetry.org | python3 -
export PATH="$HOME/.local/bin:$PATH"

# Force Poetry to ignore any active virtual environment markers
export POETRY_IGNORE_VIRTUALENVS=1

# Configure Poetry to create a new virtual environment in the project directory
poetry config virtualenvs.create true
poetry config virtualenvs.in-project true

# Remove any previously created environment and create a new one using python3.12
poetry env remove --all || true
poetry env use python3.12

poetry lock
poetry install

# Get the version of git
git_version=$(git --version | awk '{print $3}')

# Convert the version to a comparable number
# Major version gets padded normally, while minor and patch versions are three digits
version_number=$(echo "$git_version" | awk -F. '{printf "%d%03d%03d", $1, $2, $3}')

# Define the required version (2.20.0)
required_version=$(echo "2.20.0" | awk -F. '{printf "%d%03d%03d", $1, $2, $3}')

# Debugging output to ensure proper values
echo "Detected Git version: $git_version "
echo "Required Git version: >= 2.20.0"

# Compare the versions
if [[ "$version_number" -gt "$required_version" ]]; then
    echo "Git version is greater than 2.20. Performing the task..."
    # Place your task commands here
    # Install pre-commit if not already installed
    if ! command -v pre-commit &>/dev/null; then
        echo "Installing pre-commit..."
        pip install pre-commit
    fi

    pre-commit install
    pre-commit autoupdate
else
    echo "Git version is not greater than 2.20. Skipping the task."
    echo "WARNING: git version is < 2.20. Can not install pre-commit hooks"
    echo "Please upgrade git on your system and re-run the install"
fi


echo ""
echo "-----------------------"
echo "Installation complete. "
echo "-----------------------"
echo "Use \"source $setup_and_run_file\" to set up and activate the environment."
