#!/bin/bash

# Capture the start time
start_time=$(date +%s)

VERSION=0.1.3

# TDS size in GB
TDS_SIZE=0.01
TDS_SIZE_SI=0.01
TDS_SIZE_IB=0.01
THEME='both' # or 'seaice' or 'iceberg'

# Colors
GREEN='\033[0;32m'
RED='\033[0;31m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

clear

echo ""
echo "--------------------------------------------------------------------------"
echo -e "Installation script for ${BLUE}CLEV2ER Sea Ice and Iceberg (native install)${NC}"
echo "Software version: $VERSION"
echo "--------------------------------------------------------------------------"
echo ""

# Function to ensure valid input
# usage: ensure_valid_response "Y N" "Enter Y or N" ANSWER
ensure_valid_response() {
    local options="$1"
    local prompt_message="$2"
    local response_var="$3"
    while true; do
        read -p "$prompt_message" RESPONSE
        RESPONSE=${RESPONSE:-$DEFAULT_RESPONSE}
        for option in ${options//,/ }; do
            if [[ "$RESPONSE" == "$option" ]]; then
                eval $response_var="'$RESPONSE'"
                return 0
            fi
        done
        echo "Invalid response. Please enter one of: $options."
    done
}


# Function to check if sha256sum or shasum is available
check_sha_installed() {
    if command -v sha256sum &> /dev/null; then
        echo "sha256sum"
    elif command -v shasum &> /dev/null; then
        echo "shasum"
    else
        echo "none"
    fi
}

# Check if sha256sum or shasum is installed
SHA_CMD=$(check_sha_installed)
if [ "$SHA_CMD" == "none" ]; then
    echo -e "${RED}Neither sha256sum nor shasum is available. Exiting.${NC}"
    exit 1
fi

# Function to check if pv is installed
check_pv_installed() {
    if command -v pv &> /dev/null; then
        echo "yes"
    else
        echo "no"
    fi
}
# Check if pv is installed
PV_INSTALLED=$(check_pv_installed)


echo -e "${BLUE}Step 1: Select theme to install (seaice, iceberg,both [combined in one package])${NC}"
#DEFAULT_RESPONSE='both'
echo ""
ensure_valid_response "seaice iceberg both" "Enter seaice, iceberg or both :): " THEME

# Conditional assignment
if [[ $THEME == "seaice" ]]; then
    THEME_STR="Sea Ice"
    THEME_TAG="si"
    TDS_SIZE=$TDS_SIZE_SI
elif [[ $THEME == "iceberg" ]]; then
    THEME_STR="Iceberg"
    THEME_TAG="ib"
    TDS_SIZE=$TDS_SIZE_IB
elif [[ $THEME == "both" ]]; then
    THEME_STR="Sea Ice and Iceberg"
    THEME_TAG="siib"
fi

echo ""
echo -e "${GREEN}$THEME_STR selected${NC}"
echo ""


# Prompt for download directory
echo -e "${BLUE}Step 2: Download the CLEV2ER SII v$VERSION software package for $THEME_STR${NC}"
echo "-------------------------------------------------------------------"
echo -e "${BLUE}Enter the ${RED}temporary${BLUE} download directory path for the software${NC}"
echo -e "${RED}Package size : 82MB${NC}"
echo "This is where the package .tar.gz are initially downloaded to."
echo "Note this can be different to the final installation directory where they are unpacked and built."
echo "Downloaded .tar.gz files will be removed (cleaned) from this directory once installed"
read -p "Enter path [or press Enter for default of current dir $(pwd) ]: " DOWNLOAD_DIR
DOWNLOAD_DIR=$(readlink -f "${DOWNLOAD_DIR:-$(pwd)}")

# Ensure the download directory exists
if [ ! -d "$DOWNLOAD_DIR" ]; then
    echo "Directory $DOWNLOAD_DIR does not exist. Creating it..."
    mkdir -p "$DOWNLOAD_DIR"
    if [ $? -ne 0 ]; then
        echo -e "${RED}Failed to create directory $DOWNLOAD_DIR. Exiting.${NC}"
        exit 1
    fi
fi

cd $DOWNLOAD_DIR

URL="https://www.cpom.ucl.ac.uk/downloads/clev2er_sii/clev2er_sii_${THEME_TAG}-$VERSION.tar.gz"

echo ""
echo "Securely downloading CLEV2ER SII (${THEME_STR} theme) v$VERSION package to $DOWNLOAD_DIR ..."
echo ""
curl -f -S --progress-bar -O $URL
if [ $? -ne 0 ]; then
    echo -e "${RED}Failed to download $URL !${NC}"
    exit 1
fi
echo -e "${GREEN}Download completed successfully.${NC}"

echo ""
echo "Securely Downloading CLEV2ER SII v$VERSION SHA256 checksum..."
echo ""

curl -f -S --progress-bar -O ${URL}.sha256
if [ $? -ne 0 ]; then
    echo -e "${RED}Failed to download ${URL}.sha256 !${NC}"
    exit 1
fi
echo -e "${GREEN}Download of SHA256 checksum completed successfully.${NC}"

if [ "$SHA_CMD" == "sha256sum" ]; then
    if sha256sum -c clev2er_sii_${THEME_TAG}-$VERSION.tar.gz.sha256; then
        echo -e "${GREEN}SHA256 checksum verification succeeded.${NC}"
    else
        echo -e "${RED}SHA256 checksum verification failed!${NC}"
        exit 1
    fi
else
    if shasum -a 256 -c clev2er_sii_${THEME_TAG}-$VERSION.tar.gz.sha256; then
        echo -e "${GREEN}SHA256 checksum verification succeeded.${NC}"
    else
        echo -e "${RED}SHA256 checksum verification failed!${NC}"
        exit 1
    fi
fi

# Remove the checksum file
rm clev2er_sii_${THEME_TAG}-$VERSION.tar.gz.sha256

sleep 3
clear

# Prompt for installation directory
echo ""
echo ""
echo -e "${BLUE}Step 3: Unpack and build the software package${NC}"
echo "---------------------------------------------"
echo "Software will be unpacked and built in <INSTALL_DIR>/clev2er_sii-$VERSION"
echo -e "${BLUE}Enter the final installation directory path : <INSTALL_DIR> ${NC}"
read -p "Enter path [or press Enter for default of current dir $(pwd)]: " INSTALL_DIR

INSTALL_DIR=$(readlink -f "${INSTALL_DIR:-$(pwd)}")

# Ensure the installation directory exists
if [ ! -d "$INSTALL_DIR" ]; then
    echo "Directory $INSTALL_DIR does not exist. Creating it..."
    mkdir -p "$INSTALL_DIR"
    if [ $? -ne 0 ]; then
        echo -e "${RED}Failed to create directory $INSTALL_DIR. Exiting.${NC}"
        exit 1
    fi
fi

echo ""
if [ -d $INSTALL_DIR/clev2er_sii-$VERSION ]; then
    echo "Software already installed in $INSTALL_DIR/clev2er_sii-$VERSION"
    ensure_valid_response "Y N" "Replace existing software installation of v$VERSION? (Y/N): " ANSWER
    if [[ "$ANSWER" == 'N' || "$ANSWER" == 'n' ]]; then
        echo "Exiting installation as software v$VERSION already installed"
        exit 0
    fi
    echo "Removing $INSTALL_DIR/clev2er_sii-$VERSION ..."
    rm -rf $INSTALL_DIR/clev2er_sii-$VERSION
fi

# Continue with the installation process
echo " "
echo "Installing CLEV2ER SII v$VERSION to $INSTALL_DIR/clev2er_sii-$VERSION"
# Add your installation commands here

echo "Unpacking software..."

if [ "$PV_INSTALLED" == "yes" ]; then
    pv $DOWNLOAD_DIR/clev2er_sii_${THEME_TAG}-$VERSION.tar.gz | tar -xz -C $INSTALL_DIR && rm $DOWNLOAD_DIR/clev2er_sii_${THEME_TAG}-$VERSION.tar.gz
else
    tar -xzvf $DOWNLOAD_DIR/clev2er_sii_${THEME_TAG}-$VERSION.tar.gz -C $INSTALL_DIR && rm $DOWNLOAD_DIR/clev2er_sii_${THEME_TAG}-$VERSION.tar.gz
fi

cd $INSTALL_DIR/clev2er_sii-$VERSION

./install.sh $THEME_TAG

echo -e "${GREEN}GPP software installation of v$VERSION complete!${NC}"
echo ""
echo ""

read -p "To continue to the next step, press the Return key..."

#------------------------------------------------------------------------------------------------------
#   ADF/TDS Install
#------------------------------------------------------------------------------------------------------

clear
echo -e "${BLUE}Step 4: Optional TDS/ADF Installation${NC}"
echo "---------------------------------------------"

echo -e "${BLUE}Do you want to install the optional large test data set/ADF package?${NC}"
echo "** This is ${TDS_SIZE}GB in size **"
echo "This is recommended in order to run the full GPP chains unless you provide your own ADF files"
ensure_valid_response "Y N" "Y/N : " ANSWER
if [[ "$ANSWER" == 'N' || "$ANSWER" == 'n' ]]; then
    echo -e "${GREEN}Installation of the CLEV2ER Sea Ice and Iceberg GPP software completed without external TDS/ADF package${NC}"
    echo -e "${RED}Note, that for full GPP functionality, all ADF locations will be required to be configured${NC}"
    
    echo ""
    echo -e "${BLUE}Summary${NC}"
    echo "Software package installed to : $(readlink -f "${INSTALL_DIR}")/clev2er_sii-$VERSION"
    echo "TDS/ADF package not installed "
    echo ""
    echo "To setup for software use within the current shell, run: "
    echo ""
    echo -e "${BLUE}source  $(readlink -f "${INSTALL_DIR}")/clev2er_sii-$VERSION/activate.sh${NC}"
    echo ""
    exit 0
fi

echo ""
echo -e "${BLUE}Enter ${RED}temporary${NC} directory to download the large TDS/ADF package${NC} : ${TDS_SIZE}GB for version $VERSION"
echo "Note this is where the TDS package will be downloaded to. "
echo "Downloaded .tar.gz files will be cleaned from this directory after installation. "
echo "In the next step you can choose a different location to unpack the TDS."
read -p "Enter path for TDS download (or press ENTER for default: $DOWNLOAD_DIR) : " DOWNLOAD_DIR_tmp
DOWNLOAD_DIR=$(readlink -f "${DOWNLOAD_DIR_tmp:-$DOWNLOAD_DIR}")

# Ensure the download directory exists
if [ ! -d "$DOWNLOAD_DIR" ]; then
    echo "Directory $DOWNLOAD_DIR does not exist. Creating it..."
    mkdir -p "$DOWNLOAD_DIR"
    if [ $? -ne 0 ]; then
        echo -e "${RED}Failed to create directory $DOWNLOAD_DIR. Exiting.${NC}"
        exit 1
    fi
fi

echo ""
echo "Downloading TDS for version $VERSION .."
echo ""

cd $DOWNLOAD_DIR
curl -f -S --progress-bar -O \
https://www.cpom.ucl.ac.uk/downloads/clev2er_sii/testdata_external_${THEME_TAG}_$VERSION.tar.gz
if [ $? -ne 0 ]; then
    echo -e "${RED}Failed to download testdata_external_$VERSION.tar.gz!${NC}"
    exit 1
fi

echo "Downloading TDS SHA256 checksum for $VERSION .."

curl -f -S --progress-bar -O \
https://www.cpom.ucl.ac.uk/downloads/clev2er_sii/testdata_external_${THEME_TAG}_$VERSION.tar.gz.sha256
if [ $? -ne 0 ]; then
    echo -e "${RED}Failed to download testdata_external_$VERSION.tar.gz.sha256!${NC}"
    exit 1
fi
echo -e "${GREEN}TDS download successful${NC}"

echo "Verifying SHA256 checksum..."
if [ "$SHA_CMD" == "sha256sum" ]; then
    if sha256sum -c testdata_external_${THEME_TAG}_$VERSION.tar.gz.sha256; then
        echo -e "${GREEN}SHA256 checksum verification succeeded.${NC}"
    else
        echo  -e "${RED}SHA256 checksum verification failed!${NC}"
        exit 1
    fi
else
    if shasum -a 256 -c testdata_external_${THEME_TAG}_$VERSION.tar.gz.sha256; then
        echo -e "${GREEN}SHA256 checksum verification succeeded.${NC}"
    else
        echo  -e "${RED}SHA256 checksum verification failed!${NC}"
        exit 1
    fi
fi

rm testdata_external_${THEME_TAG}_$VERSION.tar.gz.sha256

# Prompt for installation directory
SOFT_INSTALL_DIR=$(readlink -f "${INSTALL_DIR}")
echo " "
echo "TDS/ADF package will be unpacked to <TDS_INSTALL_DIR>/testdata_external"
echo -e "${BLUE}Enter the TDS installation directory <TDS_INSTALL_DIR> path${NC}"
echo "Default location is within the CLEV2ER software installation directory: ${SOFT_INSTALL_DIR}/clev2er_sii-$VERSION"

read -p "Enter path [or press ENTER to accept default]: " TDS_INSTALL_DIR
TDS_INSTALL_DIR=$(readlink -f "${TDS_INSTALL_DIR:-$SOFT_INSTALL_DIR/clev2er_sii-$VERSION}")

if [ "$TDS_INSTALL_DIR" == "${SOFT_INSTALL_DIR}/clev2er_sii-$VERSION" ]; then
    TDS_IN_STANDARD_LOCATION='yes'
fi

TDS_INSTALL_DIR=$(readlink -f "$TDS_INSTALL_DIR")

# Ensure the installation directory exists
if [ ! -d "$TDS_INSTALL_DIR" ]; then
    echo "Directory $TDS_INSTALL_DIR does not exist. Creating it..."
    mkdir -p "$TDS_INSTALL_DIR"
    if [ $? -ne 0 ]; then
        echo -e "${RED}Failed to create directory $TDS_INSTALL_DIR. Exiting.${NC}"
        exit 1
    fi
fi

echo ""
if [ -d $TDS_INSTALL_DIR/testdata_external ]; then
    echo "TDS already installed in $TDS_INSTALL_DIR/testdata_external"
    ensure_valid_response "Y N" "Replace existing TDS installation? (Y/N): " ANSWER
    if [[ "$ANSWER" == 'N' || "$ANSWER" == 'n' ]]; then
        echo "Exiting installation as TDS already installed"
        echo -e "${GREEN}CLEV2ER Sea Ice and Iceberg v$VERSION Native Installation complete!${NC}"
        echo ""
        echo -e "${BLUE}Summary${NC}"
        echo "Software package installed to : $SOFT_INSTALL_DIR/clev2er_sii-$VERSION"
        echo "TDS/ADF package installed to : $TDS_INSTALL_DIR/testdata_external"
        echo ""
        echo "To setup for software use within the current shell, run: "
        echo ""
        echo -e "${BLUE}source  $SOFT_INSTALL_DIR/clev2er_sii-$VERSION/activate.sh${NC}"
        echo ""
        exit 0
    fi
fi

# Continue with the installation process
echo " "
echo "Installing TDS to $TDS_INSTALL_DIR/testdata_external"
# Add your installation commands here

echo "Unpacking TDS v$VERSION..."

if [ "$PV_INSTALLED" == "yes" ]; then
    pv $DOWNLOAD_DIR/testdata_external_${THEME_TAG}_$VERSION.tar.gz | tar -xz -C $TDS_INSTALL_DIR && rm $DOWNLOAD_DIR/testdata_external_${THEME_TAG}_$VERSION.tar.gz
else
    tar -xzf $DOWNLOAD_DIR/testdata_external_${THEME_TAG}_$VERSION.tar.gz \
-C $TDS_INSTALL_DIR && rm $DOWNLOAD_DIR/testdata_external_${THEME_TAG}_$VERSION.tar.gz
fi

if [ "$TDS_IN_STANDARD_LOCATION" != 'yes' ]; then
    echo ""
    echo "TDS is installed externally to the main GPP software in $TDS_INSTALL_DIR/testdata_external"
    echo ""
    echo "Optionally link this directory to $SOFT_INSTALL_DIR/clev2er_sii-$VERSION/testdata_external"
    ensure_valid_response "Y N" "Add soft link? (Y/N): " ANSWER
    if [[ "$ANSWER" == 'Y' || "$ANSWER" == 'y' ]]; then
        echo "Creating link from $TDS_INSTALL_DIR/testdata_external to $SOFT_INSTALL_DIR/clev2er_sii-$VERSION/testdata_external"
        if [ -d $SOFT_INSTALL_DIR/clev2er_sii-$VERSION/testdata_external ]; then
            echo -e "${RED}$SOFT_INSTALL_DIR/clev2er_sii-$VERSION/testdata_external already exists!${NC}"
            ensure_valid_response "Y N" "Remove existing directory? (Y/N): " ANSWER
            if [[ "$ANSWER" == 'Y' || "$ANSWER" == 'y' ]]; then
                rm -rf $SOFT_INSTALL_DIR/clev2er_sii-$VERSION/testdata_external
                ln -s $TDS_INSTALL_DIR/testdata_external $SOFT_INSTALL_DIR/clev2er_sii-$VERSION/testdata_external
            fi
        else
            ln -s $TDS_INSTALL_DIR/testdata_external $SOFT_INSTALL_DIR/clev2er_sii-$VERSION/testdata_external
        fi        
    fi
fi

# Capture the end time
end_time=$(date +%s)

# Calculate the duration
duration=$((end_time - start_time))

# Report the duration
echo "Script execution time: $duration seconds"

rm $SOFT_INSTALL_DIR/clev2er_sii-$VERSION/Docker*

echo -e "${GREEN}CLEV2ER Sea Ice and Iceberg v$VERSION Native Installation complete!${NC}"
echo ""
echo -e "${BLUE}Summary${NC}"
echo "Software package installed to : $SOFT_INSTALL_DIR/clev2er_sii-$VERSION"
echo "TDS/ADF package installed to : $TDS_INSTALL_DIR/testdata_external"
echo ""
echo "To setup for software use within the current shell, run: "
echo ""
echo -e "${BLUE}source  $SOFT_INSTALL_DIR/clev2er_sii-$VERSION/activate.sh${NC}"
echo ""
