"""clev2er.algorithms.iceberg.alg_setup_shared_dict.py

CLEV2ER SII algorithm module

# Description of this Algorithm's purpose

The purpose of this algorithm is to setup the structure of the shared_dict
for this specific chain

# Main init() function steps


# Main process() function steps


# Requires from config


# Requires from shared_dict


# Contribution to shared_dict

shared_dict["ku"] (dict) : working dictionary to contain ku specific parameters
shared_dict["ka"] (dict) : working dictionary to contain ka specific parameters

shared_dict["ku"]["original] (dict) : dictionary to contain ku specific parameters
                                      that are the same dimensions as the original
                                      data/ku in the L1b
shared_dict["ka"]["original] (dict) : dictionary to contain ka specific parameters
                                      that are the same dimensions as the original
                                      data/ka in the L1b

shared_dict["ku"]["valid] (dict) : dictionary to contain ku specific parameters
                                      relating to valid meaurements going forward
shared_dict["ka"]["valid] (dict) : dictionary to contain ka specific parameters
                                      relating to valid meaurements going forward


shared_dict["ku"]["valid]["num_valid"] (int) : number of valid ku measurements going forward
shared_dict["ka"]["valid]["num_valid"] (int) : number of valid ka measurements going forward

shared_dict["ku"]["valid"]["bool_mask_to_orig"] : ndarray[bool], bool mask of valid ku nadir lat/lon
shared_dict["ka"]["valid"]["bool_mask_to_orig"] : ndarray[bool], bool mask of valid ka nadir lat/lon

shared_dict["ku"]["valid"]["indices_to_orig"] : ndarray[int], indices of valid ku measurements
shared_dict["ka"]["valid"]["indices_to_orig"] : ndarray[int], indices of valid ka measurements

shared_dict["ku"]["original"]["lat_nadir"] : ndarray[float], latitude at nadir (deg E)
shared_dict["ku"]["original"]["lon_nadir"] : ndarray[float], longitude at nadir (deg N)
shared_dict["ka"]["original"]["lat_nadir"] : ndarray[float], latitude at nadir (deg E)
shared_dict["ka"]["original"]["lon_nadir"] : ndarray[float], longitude at nadir (deg N)

"""

from typing import Tuple

import numpy as np
from codetiming import Timer  # used to time the Algorithm.process() function
from netCDF4 import Dataset  # pylint:disable=no-name-in-module

from clev2er.algorithms.base.base_alg import BaseAlgorithm

# each algorithm shares some common class code, so pylint: disable=duplicate-code


class Algorithm(BaseAlgorithm):
    """CLEV2ER Algorithm class

    contains:
         .log (Logger) : log instance that must be used for all logging, set by BaseAlgorithm
         .config (dict) : configuration dictionary, set by BaseAlgorithm
         - functions that need completing:
         .init() : Algorithm initialization function (run once at start of chain)
         .process(l1b,shared_dict) : Algorithm processing function (run on every L1b file)
         .finalize() : Algorithm finalization/closure function (run after all chain
                       processing completed)

    Inherits from BaseAlgorithm which handles interaction with the chain controller run_chain.py

    """

    def init(self) -> Tuple[bool, str]:
        """Algorithm initialization function

        Add steps in this function that are run once at the beginning of the chain
        (for example loading a DEM or Mask)

        Returns:
            (bool,str) : success or failure, error string

        Test for KeyError or OSError exceptions and raise them if found
        rather than just returning (False,"error description")

        Raises:
            KeyError : for keys not found in self.config
            OSError : for any file related errors

        Note:
        - retrieve required config data from self.config dict
        - log using self.log.info(), or self.log.error() or self.log.debug()

        """
        self.alg_name = __name__
        self.log.info("Algorithm %s initializing", self.alg_name)

        # --- Add your initialization steps below here ---

        # --- End of initialization steps ---

        return (True, "")

    @Timer(name=__name__, text="", logger=None)
    def process(self, l1b: Dataset, shared_dict: dict) -> Tuple[bool, str]:
        """Main algorithm processing function, called for every L1b file

        Args:
            l1b (Dataset): input l1b file dataset (constant)
            shared_dict (dict): shared_dict data passed between algorithms. Use this dict
                                to pass algorithm results down the chain or read variables
                                set by other algorithms.

        Returns:
            Tuple : (success (bool), failure_reason (str))
            ie
            (False,'error string'), or (True,'')
            if failure_reason in (False,failure_reason) contains anywhere within it the
            str 'SKIP_OK' then it indicates an expected failure (which is not an error), and the
            L1b file should be skipped.
            The run_chain.py controller will then continue processing the next L1b file. If
            not then the return status indicates an error, which is handled according to the
            main controller configuration to either stop processing or continue
            (after logging the error)

        Note:
        - retrieve required config data from self.config dict (read-only)
        - retrieve data from other algorithms from shared_dict
        - add results,variables from this algorithm to shared_dict
        - log using self.log.info(), or self.log.error() or self.log.debug()

        """

        # This step is required to support multi-processing. Do not modify
        success, error_str = self.process_setup(l1b)
        if not success:
            return (False, error_str)

        # -------------------------------------------------------------------
        # Perform the algorithm processing, store results that need to be passed
        # \/    down the chain in the 'shared_dict' dict     \/
        # -------------------------------------------------------------------

        for band in ["ku", "ka"]:
            shared_dict[band] = {
                "valid": {},  # valid contains the current working subset of valid measurements
                "original": {},  # original contains parameters relating to the original L1b
            }

            # initialize number valid
            shared_dict[band]["valid"]["num_valid"] = 0

            # ----------------------------------------------------------------------------------
            # Test for data/ku,ka/latitude, longitude in L1b and
            # store in ku,ka/original/lat_nadir, lon_nadir
            # ----------------------------------------------------------------------------------

            data_name = shared_dict["ident"]["data_group"]  # 'data' or 'data_20' (for LR)

            for param in ["latitude", "longitude"]:
                try:
                    shared_dict[band]["original"][f"{param[:3]}_nadir"] = l1b[data_name][band][
                        param
                    ][:].data
                except (KeyError, IndexError):
                    if self.config["alg_setup_shared_dict"][f"allow_missing_l1b_{band}_group"]:
                        shared_dict[band]["original"][f"{param[:3]}_nadir"] = np.array([])
                        self.log.warning("No %s/%s/%s data found in L1b", data_name, band, param)
                    else:
                        return (False, f"No {data_name}/{band}/{param} data found in L1b")

            # ----------------------------------------------------------------------------------
            # Test validity of lat/lon against fill value and set
            #    shared_dict["ku","ka"]["valid"]["bool_mask_to_orig"]
            #    shared_dict["ku","ka"]["valid"]["indices_to_orig"]
            #    shared_dict["ku","ka"]["valid"]["num_valid"]
            # ----------------------------------------------------------------------------------
            try:
                # test if FillValue is set
                shared_dict[band]["valid"]["bool_mask_to_orig"] = (
                    l1b[data_name][band]["latitude"][:].data
                    != l1b[data_name][band]["latitude"]._FillValue  # pylint: disable=W0212
                )
                shared_dict[band]["valid"]["indices_to_orig"] = np.where(
                    shared_dict[band]["valid"]["bool_mask_to_orig"]
                )[0]
                shared_dict[band]["valid"]["num_valid"] = shared_dict[band]["valid"][
                    "indices_to_orig"
                ].size

            except (KeyError, IndexError) as exc:
                if self.config["alg_setup_shared_dict"][f"allow_missing_l1b_{band}_group"]:
                    shared_dict[band]["valid"]["bool_mask_to_orig"] = np.array([])
                    shared_dict[band]["valid"]["indices_to_orig"] = np.array([])
                else:
                    self.log.error("data/%s/latitude not found : %s", band, exc)
                    return (False, f"data/{band}/latitude parameter not found")

            # Set invalid lat/lon to Nan
            if shared_dict[band]["original"]["lat_nadir"].size > 0:
                shared_dict[band]["original"]["lat_nadir"][
                    ~shared_dict[band]["valid"]["bool_mask_to_orig"]
                ] = np.nan

                shared_dict[band]["original"]["lon_nadir"][
                    ~shared_dict[band]["valid"]["bool_mask_to_orig"]
                ] = np.nan

                # Set longitudes between 0..360E
                shared_dict[band]["original"]["lon_nadir"][
                    shared_dict[band]["valid"]["bool_mask_to_orig"]
                ] %= 360

            self.log.info(
                "Original valid %s measurements present: %d (of %d)",
                band,
                shared_dict[band]["valid"]["num_valid"],
                len(l1b[data_name][band]["latitude"][:].data),
            )

        # -------------------------------------------------------------------
        # Returns (True,'') if success
        return (success, error_str)

    def finalize(self, stage: int = 0) -> None:
        """Algorithm finalization function - called after all processing completed

          Can be used to clean up/free resources initialized in the init() function

        Args:
            stage (int, optional): this sets the stage when this function is called
                                   by the chain controller. Useful during multi-processing.
                                   Defaults to 0. Not normally used by Algorithms.
        """
        self.log.info(
            "Finalize algorithm %s called at stage %d filenum %d",
            self.alg_name,
            stage,
            self.filenum,
        )
        # ---------------------------------------------------------------------
        # Add finalization steps here \/
        # ---------------------------------------------------------------------

        # ---------------------------------------------------------------------
