"""find_by_date_search module

find all CRISTAL L1b files by date search (start and end date in datetime
format)

within base dir specified (or recursively if specified)
"""

import glob
import logging
import os
import re
from datetime import datetime
from typing import List

from clev2er.algorithms.base.base_finder import BaseFinder

# pylint: disable=R0801
# pylint: disable=too-many-instance-attributes
# pylint: disable=too-many-branches
# pylint: disable=too-many-locals

log = logging.getLogger(__name__)


def find_files_in_date_range(
    start_date: datetime, end_date: datetime, base_dir: str, recursive: bool = False
) -> List[str]:
    """Find files matching a pattern within a date range.

    Args:
        start_date (datetime): The start of the date range.
        end_date (datetime): The end of the date range.
        base_dir (str): The directory to search for files.
        recursive (bool): Whether to search subdirectories recursively.

    Returns:
        List[str]: A list of file paths matching the criteria.
    """

    # Define the pattern
    if recursive:
        pattern = os.path.join(
            base_dir, "**", "CR?_IR_1B_??__???_????????T??????_????????T??????_*.NC"
        )
    else:
        pattern = os.path.join(base_dir, "CR?_IR_1B_??__???_????????T??????_????????T??????_*.NC")

    # Regular expression to extract the datetime strings
    date_regex = re.compile(r"_([0-9]{8}T[0-9]{6})_([0-9]{8}T[0-9]{6})_")

    matching_files = []

    # Search for files matching the pattern
    for filepath in glob.glob(pattern, recursive=recursive):
        filename = os.path.basename(filepath)

        # Extract dates from the filename
        match = date_regex.search(filename)
        if match:
            file_start_str, file_end_str = match.groups()

            # Convert to datetime objects
            file_start_date = datetime.strptime(file_start_str, "%Y%m%dT%H%M%S")
            file_end_date = datetime.strptime(file_end_str, "%Y%m%dT%H%M%S")

            # Check if the file's date range overlaps with the given range
            if (file_start_date <= end_date) and (file_end_date >= start_date):
                matching_files.append(filepath)

    return matching_files


class FileFinder(BaseFinder):
    """class to find a list of CRISTAL L1b files to process by a start and end date search,
       and within specified base dir (recursively if specified)

    Args:
        BaseFinder (BaseFinder): base finder class

    """

    # The base class is initialized with:
    # def __init__(self, log: logging.Logger | None = None, config: dict = {}):

    def find_files(self, flat_search=False) -> list[str]:
        """Search for L1b file according to pattern

        Args:
            flat_search (bool) : if True only search in l1b_base_dir, else use pattern
        Returns:
            (str): list of files

        """
        file_list: List[str] = []

        if "l1b_file_finder" not in self.config:
            raise KeyError("l1b_file_finder missing from config")

        if "base_dir" not in self.config["l1b_file_finder"]:
            raise KeyError("l1b_file_finder:base_dir missing from config")

        base_dir = self.config["l1b_file_finder"]["base_dir"]

        self.log.debug("base_dir %s", base_dir)

        if not os.path.isdir(base_dir):
            raise FileNotFoundError(f"{base_dir} directory not found")

        if self.start_date and self.end_date:
            self.log.debug("file search start_date %s", self.start_date)
            self.log.debug("file search end_date %s", self.end_date)

            file_list = find_files_in_date_range(
                self.start_date, self.end_date, base_dir, recursive=not flat_search
            )

        return file_list
