"""find_by_year_month module

find all CRISTAL L1b files by start year(s) and month(s)

using pattern:
CR?_IR_1B_???_???_{year:4d}{month:02d}*.NC

within base dir specified (or recursively if specified)
"""

import glob
import logging
import os
from typing import List

from clev2er.algorithms.base.base_finder import BaseFinder

# pylint: disable=R0801
# pylint: disable=too-many-instance-attributes
# pylint: disable=too-many-branches
# pylint: disable=too-many-locals

log = logging.getLogger(__name__)


class FileFinder(BaseFinder):
    """class to find a list of CRISTAL L1b files to process in one or more
       specified years or months, and within specified base dir (recursively if
       specified)

    Args:
        BaseFinder (BaseFinder): base finder class

    In order to find files you can optionally use the following
    which are optionally set by the run_chain.py command line parameters

    Set by command line options:
        self.months  # list of months to find
        self.years   # list of years to find
    Set by config file settings:
        config["l1b_base_dir"]

    """

    # The base class is initialized with:
    # def __init__(self, log: logging.Logger | None = None, config: dict = {}):

    def find_files(self, flat_search=False) -> list[str]:
        """Search for L1b file according to pattern

        Args:
            flat_search (bool) : if True only search in l1b_base_dir, else use pattern
        Returns:
            (str): list of files

        """
        file_list: List[str] = []

        if "l1b_file_finder" not in self.config:
            raise KeyError("l1b_file_finder missing from config")

        if "base_dir" not in self.config["l1b_file_finder"]:
            raise KeyError("l1b_file_finder:base_dir missing from config")

        base_dir = self.config["l1b_file_finder"]["base_dir"]

        if not os.path.isdir(base_dir):
            raise FileNotFoundError(f"{base_dir} directory not found")

        if len(self.months) == 0:
            self.months = list(range(1, 13))

        for year in self.years:
            for month in self.months:
                search_str = f"CR?_IR_1B_???_???_{year:4d}{month:02d}*.NC"
                if flat_search:
                    files = glob.glob(f"{base_dir}/{search_str}")
                else:
                    files = glob.glob(os.path.join(base_dir, "**", search_str), recursive=True)

                if len(files) > 0:
                    file_list.extend(files)

        return file_list
