"""pytest of algorithm
clev2er.algorithms.iceberg.alg_identify_l1b_file.py

Tests the algorithm that identifies and tests the CRISTAL L1b file type

"""

import logging
import os

import pytest
from netCDF4 import Dataset  # pylint:disable=no-name-in-module

from clev2er.utils.testing.testing import initialize_algorithms

# each algorithm test shares some common class code, so pylint: disable=duplicate-code

log = logging.getLogger(__name__)


@pytest.mark.parametrize(
    "dir_path,l1b_file,valid_file_type",
    [
        (
            f'{os.environ["CLEV2ER_BASE_DIR"]}/testdata/cristal/l1b/simulated/',
            "CRA_IR_1B_HR__SIC_20280103T120129_20280103T120138_20240221T161507_0009"
            "_____________ISRD_SIR_______.NC",
            True,  # expected to pass
        ),
        # Add SAR/SARin transcoded file(s) for iceberg testing here
        # (
        #  f'{os.environ["CLEV2ER_BASE_DIR"]}/testdata/cristal/l1b/transcoded/cs2/lrm/greenland/',
        #  "CRA_IR_1B_LR__SAC_20200101T000511_20200101T000822_20250106T134608_0190_012_5100"
        #  "_____ISRD_SIR_______.NC",
        #  True,
        # ),
        (  # this is intended to fail as it is CS2 format and not CRISTAL L1b format
            f'{os.environ["CLEV2ER_BASE_DIR"]}/testdata/cs2/l1bfiles',
            "CS_OFFL_SIR_LRM_1B_20200930T231147_20200930T232634_D001.nc",
            False,  # expected to fail
        ),
    ],
)
def test_alg_identify_l1b_file(dir_path, l1b_file, valid_file_type) -> None:
    """test of clev2er.algorithms.iceberg.alg_identify_l1b_file.py"""

    chain_name = "iceberg"
    alg_name_under_test = "alg_identify_l1b_file"

    # ------------------------------------------------------------------------
    # Load chain config files
    # ------------------------------------------------------------------------

    base_dir = os.environ["CLEV2ER_BASE_DIR"]
    assert base_dir is not None

    init_flag, algorithms, _ = initialize_algorithms(chain_name, alg_name_under_test, log, None)

    assert init_flag, "initialization of algorithms failed"

    # ------------------------------------------------------------------------
    # Call the Algorithm's process() function
    # ------------------------------------------------------------------------

    # Test with L1b file

    l1b_file = f"{dir_path}/{l1b_file}"
    try:
        l1b = Dataset(l1b_file)
        log.info("Opened %s", l1b_file)
    except IOError:
        assert False, f"{l1b_file} could not be read"

    # Run  Algorithm.process()
    shared_dict: dict = {"l1b_file_name": l1b_file}

    # Run  Algorithm.process() for each algorithm up until and including  algorithm under test
    for algorithm in algorithms:
        alg_name = algorithm.alg_name.split(".")[3]  # "clev2er.algorithms.<chain_name>.<alg_name>"

        return_status, failure_reason = algorithm.process(l1b, shared_dict)
        if alg_name == alg_name_under_test:
            assert return_status == valid_file_type, failure_reason
            break
        if "SKIP_OK" in failure_reason:
            log.warning("%s returned SKIP_OK", alg_name)
        assert return_status or "SKIP_OK" in failure_reason, failure_reason

    # ------------------------------------------------------------------------
    # Call the Algorithm's finalize() functions
    # ------------------------------------------------------------------------

    # Run the algorithm finalize() functions to clean up
    for algorithm in algorithms:
        alg_name = algorithm.alg_name.split(".")[3]  # "clev2er.algorithms.<chain_name>.<alg_name>"

        # Clean up algorithm
        algorithm.finalize()

        if alg_name == alg_name_under_test:
            break

    # ------------------------------------------------------------------------
    # Test the algorithm's outputs
    # ------------------------------------------------------------------------

    assert return_status == valid_file_type, failure_reason

    if valid_file_type:
        assert "mission" in shared_dict["ident"]
        assert "instrument_mode" in shared_dict["ident"]
        assert "resolution" in shared_dict["ident"]
        assert "data_group" in shared_dict["ident"]

        assert shared_dict["ident"]["mission"] in ["CRA", "CRB"]
        assert shared_dict["ident"]["instrument_mode"] in ["SIC", "SIO", "SAC"]
        assert shared_dict["ident"]["resolution"] in ["HR_", "LR_", "FF_", "LRM", "PLR", "LMC"]
        if "L" in shared_dict["ident"]["resolution"]:
            assert shared_dict["ident"]["data_group"] == "data_20"
        else:
            assert shared_dict["ident"]["data_group"] == "data"
