"""clev2er.algorithms.seaice.alg_identify_l1b_file.py

CLEV2ER SII algorithm module

# Description of this Algorithm's purpose

Identify the L1b file and its contents from its filename

If it doesn't match the CRISTAL L1b FNC then reject the file

Extract useful fields from the file name in to shared_dict["ident"]

# Main initialization (init() function) steps/resources required

None

# Main process() function steps

    Extracts fields from the L1b file name characters
        mission = l1b_file_name[:3]  # CRA, CRB, CR_
        instrument = l1b_file_name[4:6]  # IR == IRIS
        processing_level = l1b_file_name[7:9]  # 1B == L1B
        resolution = l1b_file_name[10:13]  # HR_,LR_,FF_, LRM, PLR, LMC,LOS
        instrument_mode = l1b_file_name[14:17]  # "SAC", "SIC", "SIO"

    Checks that the fields are as expected for a CRISTAL L1b file
        mission must be CRA or CRB
        instrument must be IR
        processing_level must be 1B
        instrument_mode must be one of ["SAC", "SIC", "SIO"]
        resolution must be one of ["HR_", "LR_", "FF_", "LRM", "PLR", "LMC", "LOS"]
    If any of these are invalid, returns (False,"reason")

# Contribution to shared_dict

    shared_dict["ident"]["mission"] = CRA or CRB
    shared_dict["ident"]["resolution"] = one of "HR_", "LR_", "FF_", "LRM", "PLR", "LMC", "LOS",
    shared_dict["ident"]["instrument_mode"] = SAC,SIC,SIO
    shared_dict["ident"]"data_group"]: = "data" or "data_20" (for LR)
    shared_dict["ident"]"low_resolution"]: = True for LR_, PLR,LMC,LOS

# Requires from shared_dict

   shared_dict["l1b_file_name"]

"""

import os
from typing import Tuple

from codetiming import Timer  # used to time the Algorithm.process() function
from netCDF4 import Dataset  # pylint:disable=no-name-in-module

from clev2er.algorithms.base.base_alg import BaseAlgorithm

# each algorithm shares some common class code, so pylint: disable=duplicate-code

# pylint: disable=too-many-return-statements


class Algorithm(BaseAlgorithm):
    """CLEV2ER Algorithm class

    contains:
         .log (Logger) : log instance that must be used for all logging, set by BaseAlgorithm
         .config (dict) : configuration dictionary, set by BaseAlgorithm
         - functions that need completing:
         .init() : Algorithm initialization function (run once at start of chain)
         .process(l1b,shared_dict) : Algorithm processing function (run on every L1b file)
         .finalize() : Algorithm finalization/closure function (run after all chain
                       processing completed)

    Inherits from BaseAlgorithm which handles interaction with the chain controller run_chain.py

    """

    def init(self) -> Tuple[bool, str]:
        """Algorithm initialization function

        Add steps in this function that are run once at the beginning of the chain
        (for example loading a DEM or Mask)

        Returns:
            (bool,str) : success or failure, error string

        Test for KeyError or OSError exceptions and raise them if found
        rather than just returning (False,"error description")

        Raises:
            KeyError : for keys not found in self.config
            OSError : for any file related errors

        Note:
        - retrieve required config data from self.config dict
        - log using self.log.info(), or self.log.error() or self.log.debug()

        """
        self.alg_name = __name__
        self.log.info("Algorithm %s initializing", self.alg_name)

        # --- Add your initialization steps below here ---
        # Note exceptions will be caught by chain controller

        return (True, "")

    @Timer(name=__name__, text="", logger=None)
    def process(self, l1b: Dataset, shared_dict: dict) -> Tuple[bool, str]:
        """Main algorithm processing function, called for every L1b file

        Args:
            l1b (Dataset): input l1b file dataset (constant)
            shared_dict (dict): shared_dict data passed between algorithms. Use this dict
                                to pass algorithm results down the chain or read variables
                                set by other algorithms.

        Returns:
            Tuple : (success (bool), failure_reason (str))
            ie
            (False,'error string'), or (True,'')
            if failure_reason in (False,failure_reason) contains anywhere within it the
            str 'SKIP_OK' then it indicates an expected failure (which is not an error), and the
            L1b file should be skipped.
            The run_chain.py controller will then continue processing the next L1b file. If
            not then the return status indicates an error, which is handled according to the
            main controller configuration to either stop processing or continue
            (after logging the error)

        Note:
        - retrieve required config data from self.config dict (read-only)
        - retrieve data from other algorithms from shared_dict
        - add results,variables from this algorithm to shared_dict
        - log using self.log.info(), or self.log.error() or self.log.debug()

        """

        # This step is required to support multi-processing. Do not modify
        success, error_str = self.process_setup(l1b)
        if not success:
            return (False, error_str)

        # -------------------------------------------------------------------
        # Perform the algorithm processing, store results that need to be passed
        # \/    down the chain in the 'shared_dict' dict     \/
        # -------------------------------------------------------------------

        # Extract the filename from the L1b full path
        l1b_file_name = os.path.basename(shared_dict["l1b_file_name"])
        self.log.info("l1b_file_name=%s", l1b_file_name)

        # Extract info from the file name
        # eg CRA_IR_1B_HR__SIC_

        mission = l1b_file_name[:3]  # CRA, CRB, CR_
        instrument = l1b_file_name[4:6]  # IR == IRIS
        processing_level = l1b_file_name[7:9]  # 1B == L1B
        resolution = l1b_file_name[10:13]  # HR_,LR_,FF_, LRM, PLR, LMC,LOS
        instrument_mode = l1b_file_name[14:17]  # "SAC", "SIC", "SIO"

        self.log.debug("mission=%s", mission)
        self.log.debug("instrument=%s", instrument)
        self.log.debug("processing_level=%s", processing_level)
        self.log.debug("resolution=%s", resolution)
        self.log.debug("instrument_mode=%s", instrument_mode)

        # Check that information from the filename is expected
        # according to the File Naming Convention (FNC)
        allowed_mission_strings = ["CRA", "CRB"]
        if mission not in allowed_mission_strings:
            return (False, "L1b filename not valid: mission str must be CRA, CRB")

        if instrument != "IR":
            return (False, "L1b filename not valid: instrument str must be IR (IRIS)")

        if processing_level != "1B":
            return (False, "L1b filename not valid: processing_level str must be 1B (L1B)")

        allowed_resolutions = ["HR_", "LR_", "FF_", "LRM", "PLR", "LMC", "LOS"]
        if resolution not in allowed_resolutions:
            return (
                False,
                "L1b filename not valid: resolution str must" f" one of {allowed_resolutions}",
            )
        allowed_instrument_modes = ["SAC", "SIC", "SIO"]
        # SAC = SAR Closed Burst
        # SIC = SARIn Closed Burst
        # SIO = SARIn Open Burst
        if instrument_mode not in allowed_instrument_modes:
            return (
                False,
                (
                    "L1b filename not valid: instrument_mode str must"
                    f" one of {allowed_instrument_modes}"
                ),
            )

        data_group = "data"
        low_resolution = False
        if "L" in resolution:
            data_group = "data_20"
            low_resolution = True

        # Find if Sarin Degraded mode
        sarin_degraded = False
        self.log.warning("No method of detecting SIN degraded mode from L1b available")

        # Add to the shared_dict
        shared_dict["ident"] = {
            "mission": mission,  # CRA, CRB
            "resolution": resolution,  # HR_,LR_,FF_, LRM, PLR, LMC,
            "instrument_mode": instrument_mode,  # SAC,SIC,SIO
            "data_group": data_group,  # "data" or "data_20"
            "low_resolution": low_resolution,  # True if LR_, LRM, PLR, LMC
            "sarin_degraded": sarin_degraded,
        }

        # -------------------------------------------------------------------
        # Returns (True,'') if success
        return (success, error_str)

    def finalize(self, stage: int = 0) -> None:
        """Algorithm finalization function - called after all processing completed

          Can be used to clean up/free resources initialized in the init() function

        Args:
            stage (int, optional): this sets the stage when this function is called
                                   by the chain controller. Useful during multi-processing.
                                   Defaults to 0. Not normally used by Algorithms.
        """
        self.log.info(
            "Finalize algorithm %s called at stage %d filenum %d",
            self.alg_name,
            stage,
            self.filenum,
        )
        # ---------------------------------------------------------------------
        # Add finalization steps here \/
        # ---------------------------------------------------------------------

        # ---------------------------------------------------------------------
