"""pytest of algorithm
clev2er.algorithms.seaice.alg_load_mission_characterization.py

Tests the algorithm that loads mission characteristics from an ADF

For CI testing purposes Physical Constants file should be kept in a location within the repo,
such as $CLEV2ER_BASE_DIR/testdata/adf/common/
CRA_IR___CHDN_AX_00000000T000000_99999999T999999_20240201T000000__________________CPOM_SIR__V01.XML
If this is not the case then @pytest.mark.testable_in_repo should be removed.

Note that this location is set in the chain config file in:
<resources><mission_characterization>
<directory>$CLEV2ER_BASE_DIR/testdata/adf/common</directory>
<filename>
CR__AX_GR_CST__AX_00000000T000000_99999999T999999_20240201T000000__________________CPOM_SIR__V01.NC
</filename>

"""

import logging
import os

import numpy as np
import pytest
from netCDF4 import Dataset  # pylint:disable=no-name-in-module

from clev2er.utils.testing.testing import initialize_algorithms

# each algorithm test shares some common class code, so pylint: disable=duplicate-code

log = logging.getLogger(__name__)


# Select some CRISTAL L1b files to use for testing algorithms
@pytest.mark.parametrize(
    "l1b_file",
    [
        (
            "CRA_IR_1B_HR__SIC_20280103T120129_20280103T120138"
            "_20240221T161507_0009_____________ISRD_SIR_______.NC"
        ),
    ],
)
def test_alg_lig_load_constants(l1b_file) -> None:
    """test of clev2er.algorithms.seaice.alg_load_mission_characterization.py"""

    base_dir = os.environ["CLEV2ER_BASE_DIR"]
    assert base_dir is not None

    chain_name = "seaice"
    alg_name_under_test = "alg_load_mission_characterization"

    init_flag, algorithms, _ = initialize_algorithms(chain_name, alg_name_under_test, log, None)

    assert init_flag, "initialization of algorithms failed"

    # -------------------------------------------------------------------------
    # Test with L1b file

    l1b_file = f"{base_dir}/testdata/cristal/l1b/simulated/{l1b_file}"
    try:
        l1b = Dataset(l1b_file)
        log.info("Opened %s", l1b_file)
    except IOError:
        assert False, f"{l1b_file} could not be read"

    # Create the shared dictionary
    shared_dict: dict = {"l1b_file_name": l1b_file}

    # Run  Algorithm.process() for each algorithm
    for algorithm in algorithms:
        alg_name = algorithm.alg_name.split(".")[3]  # "clev2er.algorithms.<chain_name>.<alg_name>"

        return_status, failure_reason = algorithm.process(l1b, shared_dict)

        if "SKIP_OK" in failure_reason:
            log.warning("%s returned SKIP_OK", alg_name)
        assert return_status or "SKIP_OK" in failure_reason, failure_reason

        if alg_name == alg_name_under_test:
            log.info("Completed process() function for Algorithm under test")
            break

    # Run the algorithm finalize() functions to clean up
    for algorithm in algorithms:
        alg_name = algorithm.alg_name.split(".")[3]  # "clev2er.algorithms.<chain_name>.<alg_name>"

        # Clean up algorithm
        algorithm.finalize()

        if alg_name == alg_name_under_test:
            break

    # ------------------------------------------------------------------------
    # Test the algorithm's outputs
    # ------------------------------------------------------------------------

    # Check that the "mission_characterization" is now at a top level of shared_dict
    assert "mission_characterization" in shared_dict, "mission_characterization not in shared_dict"
    # Check that it is a dict type
    assert isinstance(shared_dict["mission_characterization"], dict)

    # Check an expected value in the mission characterization file
    assert (
        "freq_ku_chd" in shared_dict["mission_characterization"]
    ), 'freq_ku_chd not in shared_dict["mission_characterization"]'
    freq_ku_chd = shared_dict["mission_characterization"]["freq_ku_chd"]
    assert isinstance(freq_ku_chd, float)
    assert np.isclose(
        freq_ku_chd, 13500000000, atol=0.00001
    ), f"freq_ku_chd value {freq_ku_chd} is not close to 13500000000"
