"""pytest of algorithm
clev2er.algorithms.seaice.alg_load_physical_constants.py

Tests the algorithm that loads Physical Constants

For CI testing purposes Physical Constants file should be kept in a location within the repo,
such as $CLEV2ER_BASE_DIR/testdata/adf/common/
CR__AX_GR_CST__AX_00000000T000000_99999999T999999_20240201T000000__________________CPOM_SIR__V01.NC
If this is not the case then @pytest.mark.testable_in_repo should be removed.

Note that this location is set in the chain config file in:
<resources><physical_constants>
<directory>$CLEV2ER_BASE_DIR/testdata/adf/common</directory>
<filename>
CR__AX_GR_CST__AX_00000000T000000_99999999T999999_20240201T000000__________________CPOM_SIR__V01.NC
</filename>

"""

import logging
import os

import numpy as np
import pytest
from netCDF4 import Dataset  # pylint:disable=no-name-in-module

from clev2er.utils.testing.testing import initialize_algorithms

# each algorithm test shares some common class code, so pylint: disable=duplicate-code

log = logging.getLogger(__name__)


# Select some CRISTAL L1b files to use for testing algorithms
@pytest.mark.parametrize(
    "l1b_file",
    [
        (
            "CRA_IR_1B_HR__SIC_20280103T120129_20280103T120138"
            "_20240221T161507_0009_____________ISRD_SIR_______.NC"
        ),
    ],
)
def test_alg_lig_load_constants(l1b_file) -> None:
    """test of clev2er.algorithms.seaice.alg_load_physical_constants.py"""

    base_dir = os.environ["CLEV2ER_BASE_DIR"]
    assert base_dir is not None

    chain_name = "seaice"
    alg_name_under_test = "alg_load_physical_constants"

    init_flag, algorithms, _ = initialize_algorithms(chain_name, alg_name_under_test, log, None)

    assert init_flag, "initialization of algorithms failed"

    # -------------------------------------------------------------------------
    # Test with L1b file

    l1b_file = f"{base_dir}/testdata/cristal/l1b/simulated/{l1b_file}"
    try:
        l1b = Dataset(l1b_file)
        log.info("Opened %s", l1b_file)
    except IOError:
        assert False, f"{l1b_file} could not be read"

    # Create the shared dictionary
    shared_dict: dict = {"l1b_file_name": l1b_file}

    # Run  Algorithm.process() for each algorithm
    for algorithm in algorithms:
        alg_name = algorithm.alg_name.split(".")[3]  # "clev2er.algorithms.<chain_name>.<alg_name>"

        return_status, failure_reason = algorithm.process(l1b, shared_dict)

        if "SKIP_OK" in failure_reason:
            log.warning("%s returned SKIP_OK", alg_name)
        assert return_status or "SKIP_OK" in failure_reason, failure_reason

        if alg_name == alg_name_under_test:
            log.info("Completed process() function for Algorithm under test")
            break

    # Run the algorithm finalize() functions to clean up
    for algorithm in algorithms:
        alg_name = algorithm.alg_name.split(".")[3]  # "clev2er.algorithms.<chain_name>.<alg_name>"

        # Clean up algorithm
        algorithm.finalize()

        if alg_name == alg_name_under_test:
            break

    # ------------------------------------------------------------------------
    # Test the algorithm's outputs
    # ------------------------------------------------------------------------

    assert "physical_constants" in shared_dict, "physical_constants not in shared_dict"
    assert "PI" in shared_dict["physical_constants"], 'PI not in shared_dict["physical_constants"]'
    pi = shared_dict["physical_constants"]["PI"]
    assert isinstance(pi, float)
    assert np.isclose(pi, 3.14159, atol=0.00001), f"PI value {pi} is not close to 3.141519"

    assert "C" in shared_dict["physical_constants"], 'C not in shared_dict["physical_constants"]'
    speed_of_light = shared_dict["physical_constants"]["C"]
    assert isinstance(speed_of_light, float)
    assert np.isclose(
        speed_of_light, 2.99792458e8
    ), f"C value {speed_of_light} is not close to 2.99792458e8"

    assert 2.99792458e8
