"""Test of clev2er.utils.testing.testing.initialize_algorithms"""

import logging
import os

import numpy as np
import pytest
from netCDF4 import Dataset  # pylint:disable=no-name-in-module

from clev2er.utils.testing.testing import initialize_algorithms

# Get a logger instance from pytest
log = logging.getLogger(__name__)


# Select some CRISTAL L1b files to use for testing algorithms
@pytest.mark.parametrize(
    "l1b_file",
    [
        (
            "CRA_IR_1B_HR__SIC_20280103T120129_20280103T120138"
            "_20240221T161507_0009_____________ISRD_SIR_______.NC"
        ),
        (
            "CRA_IR_1B_LR__SIC_20280103T120129_20280103T120138"
            "_20240221T161507_0009_____________ISRD_SIR_______.NC"
        ),
    ],
)
def test_alg_template2(l1b_file: str) -> None:
    """pytest of clev2er.utils.testing.testing.initialize_algorithms

    Args:
        l1b_file (str) : name of L1b file in $CLEV2ER_BASE_DIR/testdata/cristal/l1b/simulated
    """

    base_dir = os.environ["CLEV2ER_BASE_DIR"]
    assert base_dir is not None

    alg_name_under_test = "alg_template2"
    chain_name = "testchain"

    init_flag, algorithms, _ = initialize_algorithms(chain_name, alg_name_under_test, log)

    assert init_flag, "initialization of algorithms failed"

    # -------------------------------------------------------------------------
    # Test with L1b file

    l1b_file = f"{base_dir}/testdata/cristal/l1b/simulated/{l1b_file}"
    try:
        l1b = Dataset(l1b_file)
        log.info("Opened %s", l1b_file)
    except IOError:
        assert False, f"{l1b_file} could not be read"

    # Create the shared dictionary
    shared_dict: dict = {"l1b_file_name": l1b_file}

    # Run  Algorithm.process() for each algorithm
    for algorithm in algorithms:
        alg_name = algorithm.alg_name.split(".")[3]  # "clev2er.algorithms.<chain_name>.<alg_name>"
        if alg_name == alg_name_under_test:
            log.info("Running process() function for Algorithm under test")
            # Can add any other mock inputs to shared_dict here

        return_status, failure_reason = algorithm.process(l1b, shared_dict)
        assert return_status, failure_reason

    # alg_template1 adds to shared_dict['dummy_parameter1'] of np.array([1, 2])
    # alg_template2 adds to shared_dict['dummy_parameter2'] of 2*shared_dict['dummy_parameter1']

    # Check that the shared_dict contains both of these
    assert np.array_equal(shared_dict["dummy_parameter1"], np.array([1, 2]))
    assert np.array_equal(shared_dict["dummy_parameter2"], np.array([2, 4]))

    # Run the algorithm finalize() functions to clean up
    for algorithm in algorithms:
        # Clean up algorithm
        algorithm.finalize()
