"""tool to create a copy of CRISTAL HR L1b file and copy all ku params to ka"""

import argparse
import shutil
import sys

from netCDF4 import Dataset  # pylint: disable=E0611


def replicate_ku_into_ka(input_nc, output_nc):
    """
    Create a copy of `input_nc` at `output_nc` and overwrite data/ka/* variables
    with data read from data/ku/*.
    """

    # 1. Make a byte-for-byte copy of the entire file
    shutil.copyfile(input_nc, output_nc)

    # 2. Open the copied file in append mode
    with Dataset(output_nc, "a") as ds:
        # Navigate to the groups
        g_ku = ds.groups["data"].groups["ku"]
        g_ka = ds.groups["data"].groups["ka"]

        # 3. Iterate through each variable in data/ku
        for var_name, var_obj_ku in g_ku.variables.items():
            if var_name in g_ka.variables:
                # 4. Overwrite data in data/ka/var_name with data from data/ku/var_name
                var_obj_ka = g_ka.variables[var_name]

                # Read the data from ku
                data_ku = var_obj_ku[...]

                # Write into ka
                var_obj_ka[...] = data_ku

                # Optional: Print confirmation
                print(f"Copied data/ku/{var_name} -> data/ka/{var_name}")
            else:
                # If there is no matching variable in ka, you can decide to skip or raise an error
                print(f"Variable '{var_name}' not present in data/ka. Skipping.")


def main(args):
    """main function of tool"""

    # ----------------------------------------------------------------------
    # Process Command Line Arguments for tool
    # ----------------------------------------------------------------------

    # initiate the command line parser
    parser = argparse.ArgumentParser(
        description=("tool to create a copy of CRISTAL HR L1b file and copy all ku params to ka")
    )

    parser.add_argument(
        "--infile",
        "-i",
        help=("input CRISTAL L1b file"),
        required=True,
    )

    parser.add_argument(
        "--outfile",
        "-o",
        help=("output modified CRISTAL L1b file"),
        required=True,
    )

    # read arguments from the command line
    args = parser.parse_args(args)

    replicate_ku_into_ka(args.infile, args.outfile)


if __name__ == "__main__":
    main(sys.argv[1:])
