"""
This module configures pytest to set the multiprocessing start method to 'spawn' before
any tests are run.
This ensures consistency in the multiprocessing behavior across different environments and prevents
RuntimeError caused by setting the context multiple times.
"""

import multiprocessing as mp


def pytest_configure(config):
    """
    Configure pytest before any tests are run. This function sets the
    multiprocessing start method to 'spawn' if it hasn't been set already.
    This is necessary to avoid RuntimeError in environments where the start
    method might be set implicitly or by other parts of the test suite.

    Parameters:
    config (pytest.Config): The pytest config object.
    """
    # Mark 'config' as used to avoid pylint warning
    _ = config

    try:
        if mp.get_start_method(allow_none=True) != "spawn":
            mp.set_start_method("spawn")
    except RuntimeError:
        # Context has already been set, which is fine.
        pass
