"""utils.config.load_config_settings.py

Common functions to load chain configuration files

load_algorithm_list(chain_name: str, baseline: str = "", version=0) -> list[str],list[str]
load_config_files(chain_name: str, baseline: str = "") -> dict:

"""

import glob
import logging
import os
from typing import Tuple

import xmltodict  # for parsing xml to python dict
from envyaml import (  # for parsing YAML files which include environment variables
    EnvYAML,
)

from clev2er.utils.xml.xml_funcs import set_xml_dict_types

# pylint: disable=too-many-statements
# pylint: disable=too-many-locals


module_log = logging.getLogger(__name__)


# pylint: disable=too-many-branches


def load_algorithm_list(
    chain_name: str,
    alg_list_file="",
    log: logging.Logger | None = None,
) -> Tuple[list, list, str, str]:
    """load algorithm and L1b finder list for specified chain

    Lists of algorithms and finder modules are either stored in XML or YML formats

    $CLEV2ER_BASE_DIR/config/algorithm_lists/<chain_name>/*.[xml,.yml]

    Search rules:

    Either an xml or a yml file (but not both) must exist in
    $CLEV2ER_BASE_DIR/config/algorithm_lists/<chain_name>/*.[xml,.yml]

    Args:
        chain_name (str): name of the chain to load

        alg_list_file (str,optional): path of algorithm list file to use. default="" which means
                                      search for one in standard locations
        log (logging.Logger, optional): log instance to use, default is None (use module loggger)
    Raises: KeyError,ValueError,OSError,NameError

    Returns:
        list[str], list[str], str,str:
            list of algorithm names,
            list of finder module names - may be empty list,
            filename of algorithm list used
            name of algorithm to set breakpoint after or '' if no breakpoint set
    """

    if log is None:
        log = module_log

    if not alg_list_file:
        base_dir = os.environ["CLEV2ER_BASE_DIR"]

        algorithm_list_dir = os.path.join(
            f"{base_dir}", "config", "algorithm_lists", f"{chain_name}"
        )

        if not os.path.isdir(algorithm_list_dir):
            raise OSError(f"Could not find algorithm list directory : {algorithm_list_dir}")

        # Support use of upper and lowercase .xml,.XML,.yml,.YML
        xml_alg_list_files = glob.glob(os.path.join(f"{algorithm_list_dir}", "*.[xX][mM][lL]"))
        yml_alg_list_files = glob.glob(os.path.join(f"{algorithm_list_dir}", "*.[yY][mM][lL]"))

        num_xml_alg_list_files = len(xml_alg_list_files)
        num_yml_alg_list_files = len(yml_alg_list_files)

        if num_xml_alg_list_files > 1:
            raise OSError(
                f"Only 1 algorithm list xml file allowed in {algorithm_list_dir},"
                f" {num_xml_alg_list_files} found"
            )
        if num_yml_alg_list_files > 1:
            raise OSError(
                f"Only 1 algorithm list yml file allowed in {algorithm_list_dir},"
                f" {num_yml_alg_list_files} found"
            )
        if num_yml_alg_list_files == 1 and num_xml_alg_list_files == 1:
            raise OSError(
                f"Both xml and yml list files found in {algorithm_list_dir},"
                f" only 1 type is allowed"
            )

        if num_xml_alg_list_files == 1:
            alg_list_file = xml_alg_list_files[0]
            log.info("Algorithm list XML file used: %s", alg_list_file)
        if num_yml_alg_list_files == 1:
            alg_list_file = yml_alg_list_files[0]
            log.info("Algorithm list YML file used: %s", alg_list_file)

    # --------------------------------------------------------------------------------
    # Read lists from file
    #
    # xml format:
    #
    # <algorithm_list>
    # <algorithms>
    #     <alg_template1>Enable</alg_template1>
    #     <alg_template2>Enable</alg_template2>
    # </algorithms>
    # <l1b_file_selectors>
    #     <module1_name>Enable</module1_name>
    # </l1b_file_selectors>
    # </algorithm_list>
    #
    # --------------------------------------------------------------------------------

    algorithm_list = []
    finder_module_list = []
    breakpoint_algname = ""

    if alg_list_file[-4:] == ".xml":
        # Load XML file
        with open(alg_list_file, "r", encoding="utf-8") as file:
            list_xml = file.read()

        # Use xmltodict to parse and convert the XML document
        try:
            xml_dict = dict(xmltodict.parse(list_xml))
        except Exception as exc:  # pylint: disable=broad-exception-caught
            raise OSError(f"{alg_list_file} xml format error : {exc}") from exc

        # Remove the root xml level as we don't need it
        algorithms_dict = xml_dict["algorithm_list"]["algorithms"]
        if "l1b_file_selectors" in xml_dict["algorithm_list"]:
            file_finders_dict = xml_dict["algorithm_list"]["l1b_file_selectors"]
        else:
            file_finders_dict = {}

        if file_finders_dict is not None:
            finder_module_list = list(file_finders_dict.keys())
            for finder_module in finder_module_list:
                if file_finders_dict[finder_module] == "Disable":
                    finder_module_list.remove(finder_module)
        algorithm_list = list(algorithms_dict.keys())
        for alg_module in algorithm_list:
            if algorithms_dict[alg_module] == "Disable":
                algorithm_list.remove(alg_module)
            if algorithms_dict[alg_module] == "BreakpointAfter":
                breakpoint_algname = alg_module

    elif alg_list_file[-4:] == ".yml":
        # Load YML file
        try:
            yml = EnvYAML(alg_list_file)  # read the YML and parse environment variables
        except ValueError as exc:
            raise ValueError(
                f"ERROR: algorithm list file {alg_list_file} has invalid"
                f"or unset environment variables : {exc}",
            ) from exc

        # Extract the algorithms list from the dictionary read from the YAML file
        try:
            algorithm_list = yml["algorithms"]
        except KeyError as exc:
            raise KeyError(
                f"ERROR: algorithm list file {alg_list_file} has missing key: {exc}",
            ) from exc

        # Extract the algorithms list from the dictionary read from the YAML file
        try:
            finder_module_list = yml["l1b_file_selectors"]
        except KeyError as exc:
            raise KeyError(
                f"ERROR: algorithm list file {alg_list_file} has missing key: {exc}",
            ) from exc

    else:
        raise NameError(
            f"Wrong file extension: {alg_list_file[-4:]} must be .yml or .xml in file "
            f": {alg_list_file}"
        )

    return algorithm_list, finder_module_list, alg_list_file, breakpoint_algname


def load_config_files(
    chain_name: str,
    main_config_file: str = "",
    chain_config_file: str = "",
) -> tuple[dict, str, str]:
    """function to load XML,or YML configuration files for a chain and return
       as a python dict

    Configuration files consist of:

    $CLEV2ER_BASE_DIR/config/main_config.xml

    +

    $CLEV2ER_BASE_DIR/config/chain_configs/{chain_name}/<a_config_file>.yml, .xml

    Args:
        chain_name (_type_, optional): _description_. Defaults to "",
        main_config_file (str, optional): _description_. Defaults to "".
        chain_config_file (str, optional): _description_. Defaults to "".
    Raises: KeyError, OSError, ValueError

    Returns:
        (dict,str,str) : config, main_config_file, chain_config_file
    """
    base_dir = os.environ["CLEV2ER_BASE_DIR"]
    log = module_log

    # --------------------------------------------
    # Load main run control XML config file
    #   $CLEV2ER_BASE_DIR/config/main_config.xml
    # --------------------------------------------

    if not main_config_file:
        config_file = os.path.join(f"{base_dir}", "config", "main_config.xml")
    else:
        config_file = main_config_file
    if not os.path.exists(config_file):
        raise OSError(f"{config_file} not found")

    with open(config_file, "r", encoding="utf-8") as file:
        config_xml = file.read()

    # Use xmltodict to parse and convert the XML document
    try:
        config = dict(xmltodict.parse(config_xml))
    except Exception as exc:  # pylint: disable=broad-exception-caught
        raise OSError(f"{config_file} xml format error : {exc}") from exc

    # Convert all str values from XML to correct types: bool, int, float, str
    # and evaluate env variables
    set_xml_dict_types(config)

    _main_config_file = config_file

    log.info("Using Main config file %s", config_file)

    # -----------------------------------------------------------
    # Find chain XML config file
    #   $CLEV2ER_BASE_DIR/config/chain_configs/<chain_name>/*.xml
    # ------------------------------------------------------------

    if chain_config_file:
        _, ext = os.path.splitext(chain_config_file)
        if ext == ".xml":
            xml_chain_config_files = [chain_config_file]
            yml_chain_config_files = []
        elif ext == ".yml":
            yml_chain_config_files = [chain_config_file]
            xml_chain_config_files = []
        else:
            raise ValueError(f"chain config file {chain_config_file} must end in .xml or .yml")
    else:
        # Find chain YML or XML config file in $CLEV2ER_BASE_DIR/config/chain_configs/<chain_name>/
        xml_chain_config_files = glob.glob(
            os.path.join(
                f"{base_dir}", "config", "chain_configs", f"{chain_name}", "*.[xX][mM][lL]"
            )
        )
        yml_chain_config_files = glob.glob(
            os.path.join(
                f"{base_dir}", "config", "chain_configs", f"{chain_name}", "*.[yY][mM][lL]"
            )
        )

    num_xml_chain_config_files = len(xml_chain_config_files)
    num_yml_chain_config_files = len(yml_chain_config_files)

    # Check we don't have more than 1 XML config file
    if num_xml_chain_config_files > 1:
        error_str = "Only allowed one XML chain config file in " + os.path.join(
            f"{base_dir}", "config", "chain_configs", f"{chain_name}"
        )
        log.error(error_str)
        raise IOError(error_str)

    # Check we don't have more than 1 YML config file
    if num_yml_chain_config_files > 1:
        error_str = "Only allowed one YML chain config file in " + os.path.join(
            f"{base_dir}", "config", "chain_configs", f"{chain_name}"
        )
        log.error(error_str)
        raise IOError(error_str)

    # Check we don't have both XML and YML config files
    if num_xml_chain_config_files > 0 and num_yml_chain_config_files > 0:
        error_str = (
            "Only allowed XML or YML chain config file in "
            + os.path.join(f"{base_dir}", "config", "chain_configs", f"{chain_name}")
            + ". Not both."
        )
        log.error(error_str)
        raise IOError(error_str)

    # Check we don't have no XML and YML config files
    if num_xml_chain_config_files == 0 and num_yml_chain_config_files == 0:
        error_str = "No XML or YML chain config file in " + os.path.join(
            f"{base_dir}", "config", "chain_configs", f"{chain_name}."
        )
        log.error(error_str)
        raise IOError(error_str)

    if num_xml_chain_config_files == 1:
        # Process XML config file

        chain_config_file = xml_chain_config_files[0]

        log.info("Using XML %s", chain_config_file)

        with open(chain_config_file, "r", encoding="utf-8") as file:
            config_xml = file.read()

        # Use xmltodict to parse and convert the XML document
        try:
            chain_config = dict(xmltodict.parse(config_xml))
        except Exception as exc:  # pylint: disable=broad-exception-caught
            raise OSError(f"{chain_config_file} xml format error : {exc}") from exc

        # Convert all str values from XML to correct types: bool, int, float, str
        # and evaluate env variables
        set_xml_dict_types(chain_config)

        # Remove the root xml level as we don't need it
        chain_config = chain_config["configuration"]

    if num_yml_chain_config_files == 1:
        # Process YML config file

        chain_config_file = yml_chain_config_files[0]

        log.info("Using YML chain config file %s", chain_config_file)

        chain_config = EnvYAML(
            chain_config_file
        ).export()  # read the YML and parse environment variables

    # override "chain" settings from chain_config if present
    if "chain" in chain_config:
        for key in chain_config["chain"]:
            if key in config["chain"]:
                config["chain"][key] = chain_config["chain"][key]
        chain_config.pop("chain", None)

    config = config | chain_config

    return config, _main_config_file, chain_config_file
