"""clev2er.utils.cristal.echo_to_bsa.echo_to_bsa.py
class to calculate echo to boresight angle
"""

import numpy as np


def calculate_echo_to_boresight_angle(phase_waveform, sample_bin, wavelength, ant_baseline):
    """
    Calculate the echo to boresight angle based on input parameters.

    Parameters:
        phase_waveform (callable): A function that returns the phase waveform (Φ) at a given
        point.
        sample_bin (int): The bin location to sample the phase waveform at.
        wavelength (float): The radar wavelength (λ) in meters.
        ant_baseline (float): The antenna baseline (b) in meters.
        float: The echo to boresight angle in radians.
    """
    # Ensure all inputs are converted to numpy arrays for uniform type
    # handling and compatibility with numpy operations.
    wavelength = np.asarray(wavelength)
    ant_baseline = np.asarray(ant_baseline)
    sample_bin = np.asarray(sample_bin)

    # Validate types
    if not isinstance(phase_waveform, np.ndarray):
        raise TypeError("phase_waveform must be a np.array.")
    if np.isnan(sample_bin).any():
        raise ValueError("NaN(s) present in filtered sample_bin.")

    # if sample bin is a numeric array of whole numbers, convert to int array
    if not np.issubdtype(sample_bin.dtype, np.number):
        raise TypeError("sample_bin must be a numeric NumPy array.")
    if not np.all(sample_bin.astype(int) == sample_bin):
        raise ValueError("sample_bin must contain only whole numbers.")

    sample_bin = sample_bin.astype(int)

    # continue type validation
    if not np.issubdtype(sample_bin.dtype, np.integer):
        raise TypeError("sample_bin must be an int or a np.array of integers.")
    if not (
        np.issubdtype(wavelength.dtype, np.floating) or np.issubdtype(wavelength.dtype, np.integer)
    ):
        raise TypeError("wavelength must be a float, integer, or a np.array of those types.")
    if not (
        np.issubdtype(ant_baseline.dtype, np.floating)
        or np.issubdtype(ant_baseline.dtype, np.integer)
    ):
        raise TypeError("ant_baseline must be a float, int, or a np.array of those types.")

    # Validate wavelength
    if wavelength <= 0:
        raise ValueError("wavelength must be a positive value.")

    # Validate antenna baseline
    if ant_baseline <= 0:
        raise ValueError("antenna baseline must be positive.")

    # Validate sample_bin (ensure it is within the correct range)
    if (sample_bin < 0).any() or (sample_bin >= phase_waveform.shape[-1]).any():
        raise ValueError("sample_bin is out of the valid range for the phase_waveform.")

    # Ensure phase_waveform is not empty
    if len(phase_waveform) == 0:
        raise ValueError("phase_waveform must not be empty.")

    # Get the phase waveform value at the sample_bin
    phi = np.take(phase_waveform, sample_bin)

    # Calculate the echo to boresight angle
    echo_to_boresight_angle = (phi * wavelength) / (2 * np.pi * ant_baseline)

    return echo_to_boresight_angle
