"""pytest of clev2er.utils.cristal.echo_to_bsa.echo_to_bsa.py

This module contains pytest test cases for validating the functionality of 
echo_to_bsa.py, which calculates the echo to boresight angle based on 
phase waveform data, sample bin, wavelength, and antenna baseline.

The tests cover various aspects of the algorithm, including valid input handling, 
edge cases, and exceptions for invalid inputs, ensuring robustness and accuracy in 
calculating the ehco to boresight angle.

# Summary of Test Function
----------------------------------------------------------------------------------------------------
# calculate_echo_to_boresight_angle: 
Calculates the echo to boresight angle based on the given parameters.

Test Functions:

- test_calculate_echo_to_bsa_valid:
  Tests the `calculate_echo_to_boresight_angle` function with valid inputs, including edge cases 
  like sample bin at the first and last index of the waveform. It ensures that the 
  echo to boresight angle is calculated correctly for various input conditions.

- test_calculate_echo_to_bsa_invalid:
  Ensures the `calculate_echo_to_boresight_angle` function raises appropriate exceptions 
  for invalid inputs such as out-of-range sample bins, negative wavelengths, zero or 
  negative antenna baselines, and invalid types for any of the parameters.

"""

import logging

import numpy as np
import pytest

from clev2er.utils.cristal.echo_to_bsa.echo_to_bsa import (
    calculate_echo_to_boresight_angle,
)

# each algorithm test shares some common class code, so pylint: disable=duplicate-code

log = logging.getLogger(__name__)


@pytest.mark.parametrize(
    "phase_waveform, sample_bin, wavelength, ant_baseline, expected_echo_to_bsa",
    [
        # Valid case: standard input values
        (
            np.array([2.5] * 100),  # phase_waveform as constant numpy array
            10,  # sample_bin within the range
            0.03,  # wavelength
            0.5,  # ant_baseline
            0.0238,  # expected echo to bsa (calculated manually or expected result)
        ),
        # Edge case: sample_bin at the first index (boundary condition)
        (
            np.array([2.5] * 100),  # phase_waveform as constant numpy array
            0,  # sample_bin at the start of the waveform (edge case)
            0.03,  # wavelength
            0.5,  # ant_baseline
            0.0239,  # expected aexpected echo to bsa
        ),
        # Edge case: sample_bin at the last index (boundary condition)
        (
            np.array([2.5] * 100),  # phase_waveform as constant numpy array
            99,  # sample_bin at the end of the waveform (edge case)
            0.03,  # wavelength
            0.5,  # ant_baseline
            0.0239,  # expected echo to bsa
        ),
    ],
)
def test_calculate_echo_to_bsa_valid(
    phase_waveform,
    sample_bin,
    wavelength,
    ant_baseline,
    expected_echo_to_bsa,
):
    """Test the calculate_echo_to_boresight_angle function with valid and edge case inputs."""

    # Call the function
    echo_to_boresight_angle = calculate_echo_to_boresight_angle(
        phase_waveform, sample_bin, wavelength, ant_baseline
    )

    # Assert the result is as expected
    assert np.isclose(echo_to_boresight_angle, expected_echo_to_bsa, atol=1e-3)


@pytest.mark.parametrize(
    "phase_waveform, sample_bin, wavelength, ant_baseline, expected_exception",
    [
        # Invalid test case: sample_bin out of index range
        (
            np.array([2.5] * 100),  # phase_waveform defined as a constant numpy array
            1000,  # sample_bin (out of range, assuming the waveform has fewer than 1000 samples)
            0.03,  # wavelength
            0.5,  # ant_baseline
            ValueError,  # Expected to raise ValueError
        ),
        # Invalid test case: negative wavelength
        (
            np.array([2.5] * 100),  # phase_waveform defined as a constant numpy array
            10,  # sample_bin (valid sample_bin)
            -0.03,  # negative wavelength (invalid)
            0.5,  # ant_baseline
            ValueError,  # Expected to raise ValueError
        ),
        # Invalid test case: zero or negative antenna baseline
        (
            np.array([2.5] * 100),  # phase_waveform defined as a constant numpy array
            10,  # sample_bin (valid sample_bin)
            0.03,  # wavelength
            0,  # zero antenna baseline (invalid)
            ValueError,  # Expected to raise ValueError
        ),
        (
            np.array([2.5] * 100),  # phase_waveform defined as a constant numpy array
            10,  # sample_bin (valid sample_bin)
            0.03,  # wavelength
            -0.5,  # negative antenna baseline (invalid)
            ValueError,  # Expected to raise ValueError
        ),
        # Invalid test case: empty phase_waveform
        (
            np.array([]),  # phase_waveform is an empty numpy array
            10,  # sample_bin (valid sample_bin)
            0.03,  # wavelength
            0.5,  # ant_baseline
            ValueError,  # Expected to raise ValueError
        ),
        # Invalid test case: invalid type for phase_waveform (string instead of array)
        (
            "invalid_type",  # phase_waveform as a string (invalid type)
            10,  # sample_bin (valid sample_bin)
            0.03,  # wavelength
            0.5,  # ant_baseline
            TypeError,  # Expected to raise TypeError
        ),
        # Invalid test case: invalid type for sample_bin (string instead of int)
        (
            np.array([2.5] * 100),  # phase_waveform defined as a constant numpy array
            "invalid_type",  # sample_bin as a string (invalid type)
            0.03,  # wavelength
            0.5,  # ant_baseline
            TypeError,  # Expected to raise TypeError
        ),
        # Invalid test case: invalid type for wavelength (string instead of float)
        (
            np.array([2.5] * 100),  # phase_waveform defined as a constant numpy array
            10,  # sample_bin (valid sample_bin)
            "invalid_type",  # wavelength as a string (invalid type)
            0.5,  # ant_baseline
            TypeError,  # Expected to raise TypeError
        ),
        # Invalid test case: invalid type for antenna baseline (string instead of float)
        (
            np.array([2.5] * 100),  # phase_waveform defined as a constant numpy array
            10,  # sample_bin (valid sample_bin)
            0.03,  # wavelength
            "invalid_type",  # ant_baseline as a string (invalid type)
            TypeError,  # Expected to raise TypeError
        ),
    ],
)
def test_calculate_echo_to_bsa_invalid(
    phase_waveform,
    sample_bin,
    wavelength,
    ant_baseline,
    expected_exception,
):
    """Test the calculate_echo_to_boresight function with invalid inputs."""

    with pytest.raises(expected_exception):
        calculate_echo_to_boresight_angle(phase_waveform, sample_bin, wavelength, ant_baseline)
