"""
# pytest of clev2er.utils.cristal.reloc_sarin.reloc_sarin.py

This module contains pytest test cases for validating the functionality of reloc_sarin.py functions.

# Summary of Test Functions
----------------------------------------------------------------------------------------------------

# calculate_crf_points

Calculates the CRF points based on input parameters such as base vector, echo-to-boresight angle,  
and radius.

Test Functions:

- test_calculate_crf_points_valid:  
  Tests the calculate_crf_points function with valid inputs, verifying that the CRF points are  
  calculated correctly for a range of base vectors and radius values.

- test_calculate_crf_points_invalid:  
  Ensures the calculate_crf_points function raises appropriate exceptions for invalid inputs,  
  including out-of-range values, incorrect vector formats, or missing parameters.

----------------------------------------------------------------------------------------------------

# compute_poca_geodetic_coords

Converts CRF points to geodetic coordinates.

Test Functions:

- test_compute_poca_geodetic_coords_valid:  
  Verifies the correct transformation of CRF points to geodetic coordinates, ensuring that the  
  conversion respects the proper coordinate system and transformations.

- test_compute_poca_geodetic_coords_invalid:  
  Ensures the compute_poca_geodetic_coords function raises appropriate exceptions for invalid or  
  incorrectly formatted CRF points, such as missing data or incompatible units.

----------------------------------------------------------------------------------------------------

# get_crf_in_efc

Calculates Satellite CRF Orientation in the ECF Frame.

Test Functions:

- test_get_crf_in_efc_valid:  
  Validates the transformation from geodetic coordinates and velocity vectors to CRF axes and the  
  EFC center of gravity, checking for correct alignment and expected output.

- test_get_crf_in_efc_invalid:  
  Ensures the get_crf_in_efc function raises appropriate exceptions for invalid geodetic 
  coordinate inputs, incorrect velocity vectors, or missing transformation parameters.
"""
import numpy as np
import pytest

import clev2er.utils.cristal.reloc_sarin.reloc_sarin as rs


@pytest.mark.parametrize(
    "base_vector_filt, echo_to_bsa_filt, r_filt, expected_crf_points",
    [
        # Standard valid inputs
        (
            np.array(
                [[-9.22e-03, -3.76e-04, -9.99e-01], [-9.23e-03, -3.96e-04, -9.99e-01]]
            ),  # base_vector_filt
            np.array([0.00924, 0.00872]),  # echo_to_bsa_filt (in radians)
            np.array([750356.0, 750329.0]),  # r_filt
            np.array(
                [[750228.091, 0.0, -13850.876], [750208.061, 0.0, -13468.154]]
            ),  # expected_crf_points
        ),
    ],
)
def test_calculate_crf_points_valid(
    base_vector_filt,
    echo_to_bsa_filt,
    r_filt,
    expected_crf_points,
):
    """Test the calculate_crf_points function with valid and edge case inputs."""

    # Call the function
    result = rs.calculate_crf_points(base_vector_filt, echo_to_bsa_filt, r_filt)

    # Assert the result is as expected
    assert np.allclose(result, expected_crf_points, atol=1e-3)


@pytest.mark.parametrize(
    "base_vector_filt, echo_to_bsa_filt, r_filt, expected_error",
    [
        # mismatched shapes
        (
            np.array([[3.0, 0.0, 4.0]]),  # base_vector_filt
            np.array([[0.5, 0.8]]),  # echo_to_bsa_filt with mismatched shape
            np.array([1.0]),  # r_filt
            ValueError,
        ),
        # non-numeric array
        (
            np.array([["a", "b"], ["c", "d"]]),  # base_vector_filt with non-numeric values
            np.array([[0.5, 0.0, 0.0]]),  # echo_to_bsa_filt
            np.array([1.0, 1.5]),  # r_filt
            ValueError,
        ),
        # baseline_vector not 3D
        (
            np.array([[1.0, 0.0]]),  # base_vector_filt with incorrect shape
            np.array([[0.5, 0.0, 0.0], [0.7, 0.0, 0.0]]),  # echo_to_bsa_filt
            np.array([1.0, 1.5]),  # r_filt
            ValueError,
        ),
        # non-array input
        (
            None,  # base_vector_filt with None
            np.array([0.5, 0.0, 0.0]),  # echo_to_bsa_filt
            np.array([1.0, 1.5]),  # r_filt
            ValueError,
        ),
    ],
)
def test_calculate_crf_points_invalid(base_vector_filt, echo_to_bsa_filt, r_filt, expected_error):
    """Test the calculate_crf_points function with invalid inputs."""

    # Call the function and assert it raises the correct error
    with pytest.raises(expected_error):
        rs.calculate_crf_points(base_vector_filt, echo_to_bsa_filt, r_filt)


@pytest.mark.parametrize(
    "crf_axes, crf_points, efc_cogs, expected_lat_poca, expected_lon_poca, expected_elev_poca",
    [
        (
            np.array(
                [
                    [[-0.075, -0.06, -0.9], [-0.3, -0.9, 0.09], [0.9, -0.4, -0.04]],  # CRF axis 1
                    [[-0.08, -0.05, -0.8], [-0.4, -0.8, 0.08], [0.8, -0.3, -0.03]],
                ]
            ),  # CRF axis 2
            np.array([[750300, 0.0, -13800], [750200, 0.0, -13400]]),  # CRF point 1  # CRF point 2
            np.array(
                [[530000, 2500000, -6610000], [520000, 2400000, -6600000]]  # ECOG 1
            ),  # ECOG 2
            np.array([-68.73445693, -70.39736411]),  # Expected latitudes in degrees
            np.array([77.930, 77.359]),  # Expected longitudes in degrees
            np.array([13825.598, 14203.426]),  # Expected elevations
        )
    ],
)
def test_compute_poca_geodetic_coords_valid(
    crf_axes,
    crf_points,
    efc_cogs,
    expected_lat_poca,
    expected_lon_poca,
    expected_elev_poca,
):
    """Test compute_poca_geodetic_coords with valid inputs."""
    # Call the function
    lat, lon, elev = rs.compute_poca_geodetic_coords(crf_axes, crf_points, efc_cogs)

    # Validate results
    np.testing.assert_almost_equal(lat, expected_lat_poca, decimal=3)
    np.testing.assert_almost_equal(lon, expected_lon_poca, decimal=3)
    np.testing.assert_almost_equal(elev, expected_elev_poca, decimal=3)


@pytest.mark.parametrize(
    "lon, lat, alt, vel_vec, expected_crf_axes, expected_efc_cog",
    [
        (
            np.array([78.0, -68.0]),  # Longitudes
            np.array([78.1, -68.5]),  # Latitudes
            np.array([750000.0, 750200.0]),  # Altitudes
            np.array([[-500.0, -7000.0, -3000.0], [-510.0, -6900.0, -2700.0]]),  # Velocity vectors
            np.array(
                [
                    [
                        [-0.043, -0.05, 0.998],
                        [-0.202, -0.978, -0.058],
                        [-0.979, 0.204, -0.032],
                    ],  # CRF Axes for the first waveform
                    [[-0.137, -0.206, -0.969], [0.34, -0.929, 0.149], [0.93, 0.309, -0.197]],
                ]
            ),  # CRF Axes for the second waveform
            np.array(
                [[306479.94, 1441874.752, 6953129.95], [981224.738, -2428616.45, -6609758.023]]
            ),  # Expected ECF COG (satellite positions)
        )
    ],
)
def test_get_crf_in_efc_valid(lon, lat, alt, vel_vec, expected_crf_axes, expected_efc_cog):
    """Test get_crf_in_efc with valid inputs."""
    # Call the function
    crf_axes, efc_cog = rs.get_crf_in_efc(lon, lat, alt, vel_vec)

    # Validate results
    np.testing.assert_almost_equal(crf_axes, expected_crf_axes, decimal=3)
    np.testing.assert_almost_equal(efc_cog, expected_efc_cog, decimal=3)


@pytest.mark.parametrize(
    "lon, lat, alt, vel_vec, expected_error",
    [
        # mismatched shapes
        (
            np.array([30.0, 45.0, 60.0]),  # lon
            np.array([10.0, -10.0]),  # lat with mismatched shape
            np.array([1000.0, 2000.0, 1500.0]),  # alt
            np.array([[0.1, 0.2, 0.3], [0.2, 0.3, 0.4], [0.3, 0.4, 0.5]]),  # vel_vec
            ValueError,
        ),
        # non-numeric input
        (
            np.array([30.0, 45.0, 60.0]),  # lon
            np.array([10.0, -10.0, "20.0"]),  # lat with non-numeric value
            np.array([1000.0, 2000.0, 1500.0]),  # alt
            np.array([[0.1, 0.2, 0.3], [0.2, 0.3, 0.4], [0.3, 0.4, 0.5]]),  # vel_vec
            TypeError,
        ),
        # vel_vec not 3D
        (
            np.array([30.0, 45.0, 60.0]),  # lon
            np.array([10.0, -10.0, 20.0]),  # lat
            np.array([1000.0, 2000.0, 1500.0]),  # alt
            np.array([[0.1, 0.2], [0.2, 0.3], [0.3, 0.4]]),  # vel_vec with incorrect shape
            ValueError,
        ),
        # non-array input
        (
            None,  # lon with non-array value
            np.array([10.0, -10.0, 20.0]),  # lat
            np.array([1000.0, 2000.0, 1500.0]),  # alt
            np.array([[0.1, 0.2, 0.3], [0.2, 0.3, 0.4], [0.3, 0.4, 0.5]]),  # vel_vec
            TypeError,
        ),
        # non-numeric in vel_vec
        (
            np.array([30.0, 45.0, 60.0]),  # lon
            np.array([10.0, -10.0, 20.0]),  # lat
            np.array([1000.0, 2000.0, 1500.0]),  # alt
            np.array(
                [["a", "b", "c"], [0.2, 0.3, 0.4], [0.3, 0.4, 0.5]]
            ),  # vel_vec with non-numeric values
            TypeError,
        ),
    ],
)
def test_get_crf_in_efc_invalid(lon, lat, alt, vel_vec, expected_error):
    """Test get_crf_in_efc function with invalid inputs."""

    # Call the function and assert it raises the correct error
    with pytest.raises(expected_error):
        rs.get_crf_in_efc(lon, lat, alt, vel_vec)


@pytest.mark.parametrize(
    "crf_axes, crf_points, efc_cogs, expected_error",
    [
        # Non-numeric values in CRF axes
        (
            np.array([[[1, 0, 0], [0, "a", 0], [0, 0, 1]]] * 2),
            np.array([[1, 2, 3], [4, 5, 6]]),
            np.array([[0, 0, 0], [1, 1, 1]]),
            ValueError,
        ),
        # Mismatched shapes
        (
            np.array([[[1, 0, 0], [0, 1, 0], [0, 0, 1]]] * 2),
            np.array([[1, 2, 3]]),  # Only one row
            np.array([[0, 0, 0], [1, 1, 1]]),
            ValueError,
        ),
        # Invalid dimensionality
        (
            np.array([[1, 0, 0], [0, 1, 0], [0, 0, 1]]),  # Missing one dimension
            np.array([[1, 2, 3], [4, 5, 6]]),
            np.array([[0, 0, 0], [1, 1, 1]]),
            ValueError,
        ),
    ],
)
def test_compute_poca_geodetic_coords_invalid(crf_axes, crf_points, efc_cogs, expected_error):
    """Test compute_poca_geodetic_coords with invalid inputs."""
    with pytest.raises(expected_error):
        rs.compute_poca_geodetic_coords(crf_axes, crf_points, efc_cogs)
