"""
Helper functions for pytest testing of CLEV2ER algorithm chains

initialize_algorithms() : Initialize all algorithms up until (and including)
                          the algorithm under test
"""

import importlib
import logging
import sys
from typing import Any, Dict

from clev2er.utils.config.load_config_settings import (
    load_algorithm_list,
    load_config_files,
)

# pylint: disable=too-many-locals


def merge_dicts_recursive(dict1: Dict[str, Any], dict2: Dict[str, Any]) -> Dict[str, Any]:
    """
    Recursively merges two dictionaries. If both dictionaries contain a key with a
    dictionary as the value, the function will recursively merge the sub-dictionaries.
    For other types, the value from `dict2` will overwrite the value in `dict1`.

    Args:
        dict1 (Dict[str, Any]): The first dictionary to merge.
        dict2 (Dict[str, Any]): The second dictionary to merge. Values from this dictionary will
            overwrite those in `dict1` if there are key conflicts (unless both values are
            dictionaries).

    Returns:
        Dict[str, Any]: A new dictionary that is the result of recursively merging
        `dict1` and `dict2`.

    Example:
        >>> dict1 = {'a': 1, 'b': {'x': 10, 'y': 20}, 'c': 3}
        >>> dict2 = {'b': {'y': 21, 'z': 30}, 'c': 4, 'd': 5}
        >>> merge_dicts_recursive(dict1, dict2)
        {'a': 1, 'b': {'x': 10, 'y': 21, 'z': 30}, 'c': 4, 'd': 5}
    """
    merged = dict1.copy()  # Start with dict1's keys and values

    for key, value in dict2.items():
        if key in merged and isinstance(merged[key], dict) and isinstance(value, dict):
            # If both dict1 and dict2 have the same key and both values are dictionaries,
            # merge them recursively
            merged[key] = merge_dicts_recursive(merged[key], value)
        else:
            # Otherwise, overwrite the value in dict1 with the value from dict2
            merged[key] = value

    return merged


def initialize_algorithms(
    chain_name: str, alg_name_under_test: str, log: logging.Logger, config_extra: dict | None = None
) -> tuple[bool, list, dict]:
    """Initialize all algorithms up until (and including) the algorithm under test

    Args:
        chain_name (str): CLEV2ER chain name
        alg_name_under_test (str): name of algorithm under test
        log (Logger.log) : logger instance to use for log messages
        config_extra (dict) : dictionary to merge with loaded chain config

    Returns:  (bool,list, dict) : False (failure), list[Algorithm], config dict
    """

    # Load algorithm list

    alg_names, _, _, _ = load_algorithm_list(chain_name)

    if alg_name_under_test not in alg_names:
        log.error(
            "alg_name_under_test: %s not in algorithm list for %s",
            alg_name_under_test,
            chain_name,
        )
        return False, [], {}

    # truncate the algorithm list to include algs up until the algorithm
    # under test.
    try:
        index = alg_names.index(alg_name_under_test)
        # Slice the list up to and including the target string
        alg_names = alg_names[: index + 1]
    except ValueError:
        log.error("%s not found in the list.", alg_name_under_test)
        return False, [], {}

    # Load the chain config file
    config, _, _ = load_config_files(chain_name)

    config["chain"]["chain_name"] = chain_name
    config["chain"]["use_multi_processing"] = False

    # Merge input config data into chain config file
    # requires a multi-level recursive merge.
    if config_extra is not None:
        merged_config = merge_dicts_recursive(config, config_extra)
        config = merged_config.copy()

    alg_object_list = []

    for alg_name in alg_names:
        #
        # --------------------------------------------------------------------
        # Dynamically import each Algorithm from the list
        # --------------------------------------------------------------------
        try:
            module = importlib.import_module(
                f"clev2er.algorithms.{config['chain']['chain_name']}.{alg_name}"
            )
        except ImportError as exc:
            log.error("Could not import algorithm %s, %s", alg_name, exc)
            return False, [], config

        # --------------------------------------------------------------------
        # Create an instance of each Algorithm,
        #   - runs its __init__(config) function
        # --------------------------------------------------------------------

        # Load/Initialize algorithm
        try:
            alg_obj = module.Algorithm(config, log)
        except (FileNotFoundError, IOError, KeyError, ValueError, OSError) as exc:
            exc_type, exc_value, exc_tb = sys.exc_info()
            if exc_tb is not None:
                filename = exc_tb.tb_frame.f_code.co_filename
                line_number = exc_tb.tb_lineno
            else:
                filename = "unknown"
                line_number = 0
            log.error(
                "Could not initialize algorithm %s due to %s:%s at [%s:line %d]",
                alg_name,
                exc_type.__name__ if exc_type is not None else "",
                exc_value,
                filename,
                line_number,
            )
            log.debug("%s", exc)
            return False, [], config
        alg_object_list.append(alg_obj)
    return True, alg_object_list, config
