#!/usr/bin/env python3
"""
Test Harness for CLEV2ER SII GPP
"""

import argparse
import os
import re
import subprocess
import sys
from collections import OrderedDict

# pylint: disable=subprocess-run-check

RESULTS_FILE = "test_results.txt"
VERIFICATION_REPORT_FILE = "verification_report.txt"
results: OrderedDict[str, str] = OrderedDict()

# Color codes
GREEN = "\033[0;32m"
RED = "\033[0;31m"
YELLOW = "\033[0;33m"
NC = "\033[0m"  # No Color
BLACK_BOLD = "\033[1;30m"


def sanitize(name: str) -> str:
    """Sanitize the test name by stripping leading and trailing whitespace.

    Args:
        name: The name to sanitize.

    Returns:
        The sanitized name.
    """
    return name.strip()


def parse_test_result(file_path: str) -> int:
    """Parse the test result from the output file.

    Args:
        file_path: Path to the file containing test results.

    Returns:
        An integer representing the test result:
        1 for passed,
        2 for failed,
        3 for not implemented.
    """
    with open(file_path, "r", encoding="utf-8") as this_file:
        for line in this_file:
            line = line.strip()
            if re.search(r"Test TC-VER-\d{3} passed!", line):
                return 1
            if re.search(r"Test TC-VER-\d{3} failed!", line):
                return 2
            if re.search(r"Test TC-VER-\d{3} not implemented!", line):
                return 3

    return 3


def remove_ansi_escape_sequences_from_file(file_path: str) -> None:
    """Remove ANSI escape sequences from a file.

    Args:
        file_path: Path to the file to clean.
    """
    ansi_escape = re.compile(
        r"""
        \x1B  # ESC
        (?:   # 7-bit C1 Fe (except CSI)
            [@-Z\\-_]
        |     # or [ for CSI, followed by a control sequence
            \[
            [0-?]*  # Parameter bytes
            [ -/]*  # Intermediate bytes
            [@-~]   # Final byte
        )
    """,
        re.VERBOSE,
    )

    with open(file_path, "r", encoding="utf-8") as this_file:
        content = this_file.read()

    cleaned_content = ansi_escape.sub("", content)

    with open(file_path, "w", encoding="utf-8") as this_file:
        this_file.write(cleaned_content)


def remove_non_ascii(text: str) -> str:
    """Remove non-ASCII characters from a string.

    Args:
        text: The text to clean.

    Returns:
        The cleaned text containing only ASCII characters.
    """
    return re.sub(r"[^\x00-\x7F]+", "", text)


def run_test(this_test_dir: str) -> None:
    """Run a test and capture its result.

    Args:
        this_test_dir: The directory containing the test script and output files.
    """
    this_test_name = os.path.basename(this_test_dir)
    this_test_name = sanitize(this_test_name)
    print(f"Running test: '{this_test_name}'")
    test_script = os.path.join(this_test_dir, "test.sh")
    output_file = os.path.join(this_test_dir, "test_output.txt")

    if os.path.isfile(test_script) and os.access(test_script, os.X_OK):
        try:
            _ = subprocess.run(f"{test_script} | tee {output_file}", shell=True)

            remove_ansi_escape_sequences_from_file(output_file)

            rval = parse_test_result(output_file)

            if rval == 1:
                results[this_test_name] = f"{this_test_name}: {GREEN}SUCCESS{NC}"
            elif rval == 2:
                results[this_test_name] = f"{this_test_name}: {RED}FAILURE{NC}"
            else:
                results[this_test_name] = f"{this_test_name}: {YELLOW}NOT IMPLEMENTED{NC}"
        except subprocess.CalledProcessError as e:
            print(f"Subprocess error: {e}")
            results[this_test_name] = f"{this_test_name}: {RED}SUBPROCESS ERROR: {e}{NC}"
        except FileNotFoundError as e:
            print(f"File not found: {e}")
            results[this_test_name] = f"{this_test_name}: {RED}FILE NOT FOUND: {e}{NC}"
        except PermissionError as e:
            print(f"Permission error: {e}")
            results[this_test_name] = f"{this_test_name}: {RED}PERMISSION ERROR: {e}{NC}"
        except OSError as e:
            print(f"OS error: {e}")
            results[this_test_name] = f"{this_test_name}: {RED}OS ERROR: {e}{NC}"
    else:
        print(f"Script not found or not executable: {test_script}")
        results[this_test_name] = (
            f"{this_test_name}: {YELLOW}test.sh not found or not executable{NC}"
        )


def extract_number_from_string(s: str) -> int:
    """Extract the first number found in a string.

    Args:
        s: The string to search.

    Returns:
        The first number found in the string, or 0 if no number is found.
    """
    match = re.search(r"\d+", s)
    if match:
        return int(match.group())
    return 0


def generate_verification_report() -> None:
    """Generate a verification report by concatenating all test output files."""
    with open(VERIFICATION_REPORT_FILE, "w", encoding="utf-8") as report:
        for this_test_dir in sorted(os.listdir("verification_tests")):
            this_full_test_dir = os.path.join("verification_tests", this_test_dir)
            if os.path.isdir(this_full_test_dir) and re.match(r"tc-ver-\d{3}", this_test_dir):
                output_file = os.path.join(this_full_test_dir, "test_output.txt")
                if os.path.isfile(output_file):
                    with open(output_file, "r", encoding="utf-8") as infile:
                        report.write(f"===== {this_test_dir} =====\n")
                        report.write(infile.read())
                        report.write("\n\n")


def normalize_test_number(test_str: str) -> str:
    """Normalize the test number by removing leading zeros.

    Args:
        test_str: The test number to normalize.

    Returns:
        The normalized test number.
    """
    return str(int(test_str))


def load_existing_results() -> None:
    """Load existing results from the results file into the results dictionary."""
    if os.path.isfile(RESULTS_FILE):
        with open(RESULTS_FILE, "r", encoding="utf-8") as this_file:
            for line in this_file:
                if ":" in line:
                    this_test_name, _ = line.split(":", 1)
                    results[this_test_name.strip()] = line.strip()


# Argument parser setup
parser = argparse.ArgumentParser(description="Run CLEV2ER SII GPP verification tests.")
group = parser.add_mutually_exclusive_group(required=True)
group.add_argument(
    "-t", "--test", type=str, help="Specify a test number to run (e.g., '3' or '003')."
)
group.add_argument("-a", "--all", action="store_true", help="Run all tests.")
group.add_argument(
    "-s", "--summary", action="store_true", help="Show summary without running tests."
)
group.add_argument(
    "-r", "--report", action="store_true", help="Generate a total verification report."
)
group.add_argument(
    "-c", "--clean", action="store_true", help="Clean (remove) all previous test results"
)
group.add_argument(
    "-d", "--delete", type=str, help="Specify a test number to delete (e.g., '3' or '003')."
)

args = parser.parse_args()

# Load existing results before processing arguments
load_existing_results()

# Main logic to handle input and run tests
if args.summary:
    pass  # Skip running tests
elif args.report:
    generate_verification_report()
    print(f"Verification report generated: {VERIFICATION_REPORT_FILE}")
    sys.exit(0)
elif args.test:
    TEST_NUMBER = normalize_test_number(args.test)
    test_dir = f"verification_tests/tc-ver-{TEST_NUMBER.zfill(3)}"
    if os.path.isdir(test_dir):
        run_test(test_dir)
    else:
        print(f"Test directory {test_dir} does not exist.")
        sys.exit(1)
elif args.all:
    for test_dir in sorted(os.listdir("verification_tests")):
        full_test_dir = os.path.join("verification_tests", test_dir)
        if os.path.isdir(full_test_dir) and re.match(r"tc-ver-\d{3}", test_dir):
            run_test(full_test_dir)
elif args.clean:
    response = input("Clean (remove) previous test results? (Y/N): ").strip().lower()
    if response == "y":
        if os.path.exists(RESULTS_FILE):
            os.remove(RESULTS_FILE)
            print(f"{RESULTS_FILE} has been removed.")
        else:
            print(f"{RESULTS_FILE} does not exist.")
    else:
        print("Nothing done")
    sys.exit(0)
elif args.delete:
    TEST_NUMBER = normalize_test_number(args.delete)
    test_dir = f"verification_tests/tc-ver-{TEST_NUMBER.zfill(3)}"
    test_name = f"tc-ver-{TEST_NUMBER.zfill(3)}"
    if test_name in results:
        del results[test_name]
        print(f"Removed results for {test_name} from results.")
    if os.path.isdir(test_dir):
        response = (
            input(f"Do you also want to delete the test directory {test_dir}? (Y/N): ")
            .strip()
            .lower()
        )
        if response == "y":
            subprocess.run(f"rm -rf {test_dir}", shell=True, check=True)
            print(f"Removed directory {test_dir}.")
    else:
        print(f"Test directory {test_dir} does not exist.")

if not args.summary:
    with open(RESULTS_FILE, "w", encoding="utf-8") as file:
        for test_name in sorted(results.keys(), key=extract_number_from_string):
            file.write(results[test_name] + "\n")

    print("\nTest Results:")
    with open(RESULTS_FILE, "r", encoding="utf-8") as file:
        print(file.read())

total_tests = len(results)
successes = sum(1 for result in results.values() if "SUCCESS" in result)
failures = sum(1 for result in results.values() if "FAILURE" in result)
not_implemented = sum(1 for result in results.values() if "NOT IMPLEMENTED" in result)

failed_tests = [name for name, result in results.items() if "FAILURE" in result]
not_implemented_tests = [name for name, result in results.items() if "NOT IMPLEMENTED" in result]
successes_tests = [name for name, result in results.items() if "SUCCESS" in result]

print(f"\n{BLACK_BOLD}Summary:{NC}")
print(f"Total verification tests: {total_tests}")
print(f"{GREEN}Successes: {successes}{NC}")
print(f"{RED}Failures: {failures}{NC}")
print(f"{YELLOW}Not Implemented: {not_implemented}{NC}")

print(f"\n{BLACK_BOLD}Test Numbers{NC}")
if not_implemented > 0:
    print(f"{YELLOW}Not Implemented:{NC} {' '.join(not_implemented_tests)}")
if failures > 0:
    print(f"{RED}Failed:{NC} {' '.join(failed_tests)}")
if successes > 0:
    print(f"{GREEN}Successful:{NC} {' '.join(successes_tests)}")
