#!/bin/bash

#--------------------------------------------------------------------------------------------------------
# Verification test ID: TC-VER-003
#--------------------------------------------------------------------------------------------------------

TEST_ID="TC-VER-003"

# Start time
start_time=$(date +%s)

#--------------------------------------------------------------------------------------------------------
# Case tested:
#--------------------------------------------------------------------------------------------------------
# The L2 GPP shall stop processing and display an informative result if a mandatory input file is missing.

#--------------------------------------------------------------------------------------------------------
# Pass Criteria:
#--------------------------------------------------------------------------------------------------------
# Processing stops if mandatory inputs are missing.

#--------------------------------------------------------------------------------------------------------
# Method used:
#--------------------------------------------------------------------------------------------------------
# Run the seaice chain with a single L1b file with and without a mandatory ADF
#
# mandatory ADF chosen is : physical constants ADF
# as defined in chain configuration XML file as
#   <resources>
# 	    <physical_constants>
# 		    <directory>/some/dir</directory>
# 		    <filename>CR__AX_GR_CST__AX_00000000T000000_99999999T999999_20240201T000000__________________CPOM_SIR__V01.NC</filename>
#           <mandatory>True</mandatory>
#
# We can modify the directory or filename using --conf_opts command line option (instead
# of actually changing the XML)

#--------------------------------------------------------------------------------------------------------
# Inputs:
#--------------------------------------------------------------------------------------------------------
# Sample L1b file: 
#    CRA_IR_1B_HR__SIC_20280103T120129_20280103T120138_20240221T161507_0009_____________ISRD_SIR_______.NC
# Mandatory ADF used: 
#    CR__AX_GR_CST__AX_00000000T000000_99999999T999999_20240201T000000__________________CPOM_SIR__V01.NC
# 

#--------------------------------------------------------------------------------------------------------
# Expected Outputs
#--------------------------------------------------------------------------------------------------------
#
# display an informative result if a mandatory input file is missing

#--------------------------------------------------------------------------------------------------------
# Test Processing
#--------------------------------------------------------------------------------------------------------

# Setup some colors for test clarity
GREEN='\033[0;32m'
RED='\033[0;31m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
BLACK_BOLD='\033[1;30m'
NC='\033[0m' # No Color

# Function to ensure valid input
# usage: ensure_valid_response "Y N" "Enter Y or N" ANSWER
ensure_valid_response() {
    local options="$1"
    local prompt_message="$2"
    local response_var="$3"
    while true; do
        read -p "$prompt_message" RESPONSE
        RESPONSE=${RESPONSE:-$DEFAULT_RESPONSE}
        for option in ${options//,/ }; do
            if [[ "$RESPONSE" == "$option" ]]; then
                eval $response_var="'$RESPONSE'"
                return 0
            fi
        done
        echo "Invalid response. Please enter one of: $options."
    done
}

clear
echo -e "${BLUE}TEST ID: $TEST_ID${NC}\n"

echo "GPP version used: "
run_chain -v

echo ""

# --------------------------------------------
# perform test here and return correct status
# --------------------------------------------
echo -e "\nTest processing started at $(date)"
echo "---------------------------------------"

echo -e "${BLACK_BOLD}Purpose: ${BLUE}Run chain ${RED}without${BLUE} mandatory ADF present ${NC}"
echo
echo "Here we change the directory of the resources:physical_constants file in the chain configuration"
echo "to an invalid path, so that the ADF is missing. We filter the output of the GPP for ERROR messages "
echo "for clarity of this test."

cmd="run_chain -n seaice -de \
-co resources:physical_constants:directory:/xxxxx \
-f $CLEV2ER_BASE_DIR/testdata/cristal/l1b/\
simulated/CRA_IR_1B_HR__SIO_20280103T120129_20280103T120132_20240222T143131_0003_____________ISRD_SIR_______.NC \
2>&1 | grep ERROR"



echo -e "\n${GREEN}$cmd${NC}\n"
read -p "Press ENTER to run this GPP command and view its output.."

echo -e "${BLACK_BOLD}Executing GPP command...${NC}"
eval "$cmd"
echo -e "${BLACK_BOLD}Command completed...${NC}"

echo -e "\n${BLUE}Does this display an informative result if a mandatory input file is missing?${NC}" 

ensure_valid_response "Y N" "[Y/N] : " ANSWER
echo ""
if [[ $ANSWER == 'Y' ]]; then
    test_return_status=0 # pass
else
    test_return_status=1 # fail
fi

echo "Test processing ended at $(date)"
echo "------------------------------------------------------"

# End time
end_time=$(date +%s)

# Calculate execution time
execution_time=$((end_time - start_time))

# Print execution time
echo "Execution time: $execution_time seconds"

if [ $test_return_status -eq 0 ]; then
    echo -e "${GREEN}Test $TEST_ID passed!${NC}"
elif [ $test_return_status -eq 1 ]; then
    echo -e "${RED}Test $TEST_ID failed!${NC}"
elif [ $test_return_status -eq 2 ]; then
    echo -e "${YELLOW}Test $TEST_ID not implemented!${NC}"
fi

exit $test_return_status


