#!/bin/bash

#--------------------------------------------------------------------------------------------------------
# Verification test ID: TC-VER-011
#--------------------------------------------------------------------------------------------------------

TEST_ID="TC-VER-011"

# Start time
start_time=$(date +%s)

#--------------------------------------------------------------------------------------------------------
# Case tested:
#--------------------------------------------------------------------------------------------------------

# The Level-2 GPP shall have the capability to limit the data processing to a time range that can be 
# specified using run-time configuration.

#--------------------------------------------------------------------------------------------------------
# Method used:
#--------------------------------------------------------------------------------------------------------
# Run the seaice GPP using the time/date selection command line arguments on a restricted set of
# L1b files covering different time periods, and show that the correct L1b files are selected for processing.
#
#--------------------------------------------------------------------------------------------------------
# Pass Criteria
#--------------------------------------------------------------------------------------------------------

# When a time range is specified in the command line, L2 GPP executes successfully.

#--------------------------------------------------------------------------------------------------------
# Inputs:
#--------------------------------------------------------------------------------------------------------

# CRA_IR_1B_HR__SIC_20280103T120129_20280103T120138_20240221T161507_0009_____________ISRD_SIR_______.NC
# CRA_IR_1B_HR__SIC_20280104T120129_20280104T120138_20240221T161507_0009_____________ISRD_SIR_______.NC


#--------------------------------------------------------------------------------------------------------
# Expected Outputs
#--------------------------------------------------------------------------------------------------------

# Case 1: time/date range: 2027 (all of), Expected: 0 file selected
# Case 2: time/date range: 2028-01-02 to 2028-01-03: Expected : 1 file selected (20280103T120129_20280103T120138)
# Case 3: time/date range: Jan 2028, Expected: 2 files selected (all)
# (20280103T120129_20280103T120138)
# (20280104T120129_20280104T120138)

#--------------------------------------------------------------------------------------------------------
# Test Processing
#--------------------------------------------------------------------------------------------------------

# Setup some colors for test clarity
GREEN='\033[0;32m'
RED='\033[0;31m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
BLACK_BOLD='\033[1;30m'
NC='\033[0m' # No Color

# Function to ensure valid input
# usage: ensure_valid_response "Y N" "Enter Y or N" ANSWER
ensure_valid_response() {
    local options="$1"
    local prompt_message="$2"
    local response_var="$3"
    while true; do
        read -p "$prompt_message" RESPONSE
        RESPONSE=${RESPONSE:-$DEFAULT_RESPONSE}
        for option in ${options//,/ }; do
            if [[ "$RESPONSE" == "$option" ]]; then
                eval $response_var="'$RESPONSE'"
                return 0
            fi
        done
        echo "Invalid response. Please enter one of: $options."
    done
}


echo -e "${BLUE}TEST ID: $TEST_ID${NC}\n"

echo "GPP version used: "
run_chain -v

echo ""

# --------------------------------------------
# perform test here and return correct status
# --------------------------------------------
echo "------------------------------------------------------"
echo -e "Test processing started at $(date)"
echo "------------------------------------------------------"
echo 

# Get the full path to the script, resolving symlinks if necessary
SCRIPT_PATH=$(readlink -f "$0")

# Get the directory containing the script
SCRIPT_DIR=$(dirname "$SCRIPT_PATH")

echo "L1b Files in a test dir from different dates"
echo
ls $SCRIPT_DIR/testdata

# CRA_IR_1B_HR__SIC_20280103T120129_20280103T120138_20240221T161507_0009_____________ISRD_SIR_______.NC
# CRA_IR_1B_HR__SIC_20280104T120129_20280104T120138_20240221T161507_0009_____________ISRD_SIR_______.NC

#----------------------------------------------------------------------------------
# Test 1: Process files outside test data's date range of 20280103, 20280104
#       : Try any date in 2027
#----------------------------------------------------------------------------------

echo
echo "Lets try processing only files from date range: 2027 (there should be no files in this range)"
echo

cmd="run_chain -n seaice -r -d $SCRIPT_DIR/testdata --year 2027  2>&1 | awk '/seaice chain processed/ || /l1b_file_name=/ || /ERROR/' "

echo -e "${BLACK_BOLD}GPP test command:${NC}"
echo -e "\n${GREEN}$cmd${NC}\n" 

read -p "Press ENTER to run this GPP command and view its output (filtered for just ERROR messages).."  
echo -e "${BLACK_BOLD}Executing GPP command...${NC}"
eval "$cmd"
echo -e "${BLACK_BOLD}Command completed...${NC}"

echo -e "\n${BLUE}Were there 0 files processed as expected?${NC}" 

ensure_valid_response "Y N" "[Y/N] : " ANSWER
echo ""

if [[ $ANSWER == 'N' ]]; then
    echo -e "${RED}Test $TEST_ID failed!${NC}"
    # Print execution time
    echo "Execution time: $(($(date +%s) - start_time)) seconds"
    exit 1
fi

#----------------------------------------------------------------------------------
# Test 2: Process files within a specific date range: 2028-01-02 to 2028-01-03
#----------------------------------------------------------------------------------

echo
echo "Lets try processing only files from date range: 2028-01-02 to 2028-01-03 (expected 1 file)"
echo


cmd="run_chain -n seaice -r -d $SCRIPT_DIR/testdata --start_date 2028-01-02 --end_date 2028-01-03 2>&1 | awk '/seaice chain processed/ || /l1b_file_name=/ || /ERROR/'"

echo -e "${BLACK_BOLD}GPP test command:${NC}"
echo -e "\n${GREEN}${cmd}${NC}\n" 

read -p "Press ENTER to run this GPP command and view its output.." input_var

echo -e "${BLACK_BOLD}Executing GPP command...${NC}"
eval "$cmd"
echo -e "${BLACK_BOLD}Command completed...${NC}"

echo -e "\n${BLUE}Does this correctly process only 1 file in the time range 2028-01-02 to 2028-01-03?${NC}" 

ensure_valid_response "Y N" "[Y/N] : " ANSWER
echo ""
if [[ $ANSWER == 'N' ]]; then
    echo -e "${RED}Test $TEST_ID failed!${NC}"
    # Print execution time
    echo "Execution time: $(($(date +%s) - start_time)) seconds"
    exit 1
fi

#----------------------------------------------------------------------------------
# Test 3: Process files within a specific month: Jan 2028
#----------------------------------------------------------------------------------

echo
echo "Lets try processing only files from date range: Jan 2028"
echo

cmd="run_chain -n seaice -r -d $SCRIPT_DIR/testdata --year 2028 --month 1 2>&1 | awk '/seaice chain processed/ || /l1b_file_name=/ || /ERROR/' "

echo -e "${BLACK_BOLD}GPP test command:${NC}"
echo -e "\n${GREEN}$cmd${NC}\n" 

read -p "Press ENTER to run this GPP command and view its output.." input_var

echo -e "${BLACK_BOLD}Executing GPP command...${NC}"
eval "$cmd"
echo -e "${BLACK_BOLD}Command completed...${NC}"

echo -e "\n${BLUE}Does this correctly process 2 files in the time range of Jan 2028?${NC}" 

ensure_valid_response "Y N" "[Y/N] : " ANSWER
echo ""
if [[ $ANSWER == 'N' ]]; then
    echo -e "${RED}Test $TEST_ID failed!${NC}"
    # Print execution time
    echo "Execution time: $(($(date +%s) - start_time)) seconds"
    exit 1
fi


echo "------------------------------------------------------"
echo "Test processing ended at $(date)"
echo "------------------------------------------------------"

echo "Execution time: $(($(date +%s) - start_time)) seconds"


echo -e "${GREEN}Test $TEST_ID passed!${NC}"

exit 0
